/*
 * Copyright (c) 2019. NUM Technology Ltd
 */

package uk.num.numlib.internal.module;

import lombok.Getter;
import lombok.NonNull;
import lombok.extern.log4j.Log4j2;
import uk.num.numlib.exc.NumBadURLException;
import uk.num.numlib.exc.NumInvalidParameterException;
import uk.num.numlib.exc.NumInvalidRedirectException;
import uk.num.numlib.internal.ctx.AppContext;
import uk.num.numlib.internal.util.*;

import java.net.MalformedURLException;
import java.util.Arrays;

/**
 * Class to hold the DNS query strings for a module and NUM ID combination.
 *
 * @author tonywalmsley
 */
@Log4j2
public class ModuleDNSQueries {
    /**
     * The module ID, e.g. "1"
     */
    @Getter
    @NonNull
    private final String moduleId;

    /**
     * The NUM ID to be queried.
     */
    @NonNull
    private final String numId;

    /**
     * The independent record query
     */
    @Getter
    private String independentRecordLocation;

    /**
     * The root independent record query - used when generating redirects
     */
    @Getter
    private String rootIndependentRecordLocation;

    /**
     * The hosted record query.
     */
    @Getter
    private String hostedRecordLocation;

    /**
     * The root hosted record query - used when generating redirects.
     */
    @Getter
    private String rootHostedRecordLocation;

    /**
     * The populator query
     */
    @Getter
    private String populatorLocation;

    /**
     * The root/branch query flag.
     */
    @Getter
    private boolean rootQuery = true;

    /**
     * Constructor
     *
     * @param moduleId the module ID string
     * @param numId    the NUM ID.
     */
    public ModuleDNSQueries(@NonNull final NonBlankString moduleId, @NonNull final NonBlankString numId) {
        log.debug("ModuleDNSQueries({}, {})", moduleId, numId);
        this.moduleId = moduleId.value;
        this.numId = numId.value;
    }

    /**
     * Build the DNS query Strings and set the root/branch flag.
     *
     * @param appContext An AppContext object
     * @throws NumBadURLException           on error
     * @throws NumInvalidParameterException on error
     */
    public void initialise(final AppContext appContext) throws NumBadURLException, NumInvalidParameterException {
        log.trace("initialise()");

        // Create a suitable LookupGenerator based on the type of the record specifier
        final LookupGenerator lookupGenerator;
        if (numId.contains("@")) {
            lookupGenerator = new EmailLookupGenerator(appContext, numId);
        } else if (numId.startsWith("http")) {
            try {
                lookupGenerator = new URLLookupGenerator(appContext, numId);
            } catch (final MalformedURLException e) {
                throw new NumBadURLException(e);
            }
        } else {
            lookupGenerator = new DomainLookupGenerator(appContext, numId);
        }

        independentRecordLocation = lookupGenerator.getIndependentLocation(moduleId);
        rootIndependentRecordLocation = lookupGenerator.getRootIndependentLocation(moduleId);
        hostedRecordLocation = lookupGenerator.getHostedLocation(moduleId);
        rootHostedRecordLocation = lookupGenerator.getRootHostedLocation(moduleId);
        rootQuery = lookupGenerator.isDomainRoot();
        if (rootQuery) {
            populatorLocation = lookupGenerator.getPopulatorLocation(moduleId);
        }
    }


    /**
     * Change the independent record location due to a redirect.
     *
     * @param appContext the AppContext
     * @param redirectTo the root-relative branch record to use.
     * @throws NumInvalidParameterException on error
     * @throws NumBadURLException           on error
     * @throws NumInvalidRedirectException  on error
     */
    public void setRootRedirectIndependentRecordLocation(final AppContext appContext, final NonBlankString redirectTo) throws
                                                                                                                       NumInvalidParameterException,
                                                                                                                       NumBadURLException,
                                                                                                                       NumInvalidRedirectException {
        final String newLocation = redirectTo.value + StringConstants.DOMAIN_SEPARATOR + rootIndependentRecordLocation;
        if (independentRecordLocation.equals(newLocation)) {
            log.debug("Attempted redirect to same location: {}", newLocation);
            throw new NumInvalidRedirectException("Cannot redirect back to the same location.");
        }
        this.independentRecordLocation = newLocation;
    }

    /**
     * Change the independent record location due to a redirect.
     *
     * @param appContext the AppContext
     * @param redirect   the relative branch record to use.
     * @param levels     the number of levels to go up before adding the redirect location
     * @throws NumInvalidRedirectException  on error
     * @throws NumInvalidParameterException on error
     * @throws NumBadURLException           on error
     */
    public void setRelativeRedirectIndependentRecordLocation(final AppContext appContext, final NonBlankString redirect, final int levels) throws
                                                                                                                                           NumInvalidRedirectException,
                                                                                                                                           NumInvalidParameterException,
                                                                                                                                           NumBadURLException {
        final String[] split = independentRecordLocation.split("\\.");
        if (levels < split.length) {
            final String[] parts = Arrays.copyOfRange(split, levels, split.length);
            final String ancestor = String.join(StringConstants.DOMAIN_SEPARATOR, parts);
            final String newLocation = redirect + StringConstants.DOMAIN_SEPARATOR + ancestor + StringConstants.DOMAIN_SEPARATOR;
            if (!newLocation.endsWith(rootIndependentRecordLocation)) {
                throw new NumInvalidRedirectException("Cannot redirect " + independentRecordLocation + " using " + redirect + " and " + levels + " levels");
            }
            if (independentRecordLocation.equals(newLocation)) {
                log.debug("Attempted redirect to same location: {}", newLocation);
                throw new NumInvalidRedirectException("Cannot redirect back to the same location.");
            }
            independentRecordLocation = newLocation;
        } else {
            throw new NumInvalidRedirectException("Cannot redirect " + independentRecordLocation + " using " + redirect + " and " + levels + " levels");
        }
    }


    /**
     * Change the hosted record location due to a redirect.
     *
     * @param appContext the AppContext
     * @param redirect   the root-relative branch record to use.
     * @throws NumInvalidParameterException on error
     * @throws NumBadURLException           on error
     * @throws NumInvalidRedirectException  on error
     */
    public void setRootRedirectHostedRecordLocation(final AppContext appContext, final NonBlankString redirect) throws
                                                                                                                NumInvalidParameterException,
                                                                                                                NumBadURLException,
                                                                                                                NumInvalidRedirectException {
        final String newLocation = redirect.value + StringConstants.DOMAIN_SEPARATOR + rootHostedRecordLocation;
        if (hostedRecordLocation.equals(newLocation)) {
            log.debug("Attempted redirect to same location: {}", newLocation);
            throw new NumInvalidRedirectException("Cannot redirect back to the same location.");
        }
        hostedRecordLocation = newLocation;
    }

    /**
     * Change the hosted record location due to a redirect.
     *
     * @param appContext the AppContext
     * @param redirect   the relative branch record to use.
     * @param levels     the number of levels to go up before adding the redirect location
     * @throws NumInvalidRedirectException  on error
     * @throws NumInvalidParameterException on error
     * @throws NumBadURLException           on error
     */
    public void setRelativeRedirectHostedRecordLocation(final AppContext appContext, final NonBlankString redirect, final int levels) throws
                                                                                                                                      NumInvalidRedirectException,
                                                                                                                                      NumInvalidParameterException,
                                                                                                                                      NumBadURLException {
        final String[] split = hostedRecordLocation.split("\\.");
        if (levels < split.length) {
            final String[] parts = Arrays.copyOfRange(split, levels, split.length);
            final String ancestor = String.join(StringConstants.DOMAIN_SEPARATOR, parts);
            final String newLocation = redirect + StringConstants.DOMAIN_SEPARATOR + ancestor + StringConstants.DOMAIN_SEPARATOR;
            if (!newLocation.endsWith(rootHostedRecordLocation)) {
                throw new NumInvalidRedirectException("Cannot redirect " + hostedRecordLocation + " using " + redirect + " and " + levels + " levels");
            }

            if (hostedRecordLocation.equals(newLocation)) {
                log.debug("Attempted redirect to same location: {}", newLocation);
                throw new NumInvalidRedirectException("Cannot redirect back to the same location.");
            }
            hostedRecordLocation = newLocation;
        } else {
            throw new NumInvalidRedirectException("Cannot redirect " + hostedRecordLocation + " using " + redirect + " and " + levels + " levels");
        }
    }

    /**
     * Handle simple redirects
     *
     * @param redirectTo the target
     */
    public void redirectIndependentRecordLocationRelativeToCurrent(final NonBlankString redirectTo) {
        independentRecordLocation = redirectTo + StringConstants.DOMAIN_SEPARATOR + independentRecordLocation;
    }

    /**
     * Handle simple redirects
     *
     * @param redirectTo the target
     */
    public void redirectHostedRecordLocationRelativeToCurrent(final NonBlankString redirectTo) {
        hostedRecordLocation = redirectTo + StringConstants.DOMAIN_SEPARATOR + hostedRecordLocation;
    }

    /**
     * A Zone Distribution Record has been found so we need to update the email lookups accordingly.
     *
     * @param appContext the AppContext
     * @param levels     the number of levels to use for zone distribution
     */
    public void setEmailRecordDistributionLevels(final AppContext appContext, final int levels) {
        if (numId.contains("@")) {
            // This only applies to email NUM IDs
            final EmailLookupGenerator generator = new EmailLookupGenerator(appContext, numId);
            independentRecordLocation = generator.getDistributedIndependentLocation(moduleId, levels);
            hostedRecordLocation = generator.getDistributedHostedLocation(moduleId, levels);
        } else {
            log.warn("Attempt to distribute a non-email lookup using a Zone Distribution Record.");
        }
    }
}
