/*
 * Decompiled with CFR 0.152.
 */
package tech.ydb.table.impl.pool;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.util.Iterator;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiConsumer;
import javax.annotation.concurrent.ThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tech.ydb.core.Status;
import tech.ydb.core.StatusCode;
import tech.ydb.core.UnexpectedResultException;

@ThreadSafe
public class WaitingQueue<T>
implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(WaitingQueue.class);
    @VisibleForTesting
    static final int WAITINGS_LIMIT_FACTOR = 10;
    private final Handler<T> handler;
    private volatile Limits limits;
    private volatile boolean stopped;
    private final ConcurrentLinkedDeque<T> idle = new ConcurrentLinkedDeque();
    private final Map<T, T> used = new ConcurrentHashMap<T, T>();
    private final Map<CompletableFuture<T>, CompletableFuture<T>> pendingRequests = new ConcurrentHashMap<CompletableFuture<T>, CompletableFuture<T>>();
    private final AtomicInteger queueSize = new AtomicInteger();
    private final Queue<CompletableFuture<T>> waitingAcquires = new ConcurrentLinkedQueue<CompletableFuture<T>>();
    private final AtomicInteger waitingAcqueireCount = new AtomicInteger();

    @VisibleForTesting
    WaitingQueue(Handler<T> handler, int maxSize, int waitingsLimit) {
        Preconditions.checkArgument((maxSize > 0 ? 1 : 0) != 0, (String)"WaitingQueue max size (%d) must be positive", (int)maxSize);
        Preconditions.checkArgument((handler != null ? 1 : 0) != 0, (Object)"WaitingQueue handler must be not null");
        this.handler = handler;
        this.limits = new Limits(maxSize, waitingsLimit);
    }

    public WaitingQueue(Handler<T> handler, int maxSize) {
        this(handler, maxSize, maxSize * 10);
    }

    public void updateLimits(int maxSize) {
        this.updateLimits(maxSize, maxSize * 10);
    }

    public void updateLimits(int maxSize, int waitingsLimit) {
        this.limits = new Limits(maxSize, waitingsLimit);
        this.checkNextWaitingAcquire();
    }

    public void acquire(CompletableFuture<T> acquire) {
        boolean ok;
        if (this.stopped) {
            acquire.completeExceptionally(new IllegalStateException("Queue is already closed"));
            return;
        }
        boolean bl = ok = this.tryToPollIdle(acquire) || this.tryToCreateNewPending(acquire) || this.tryToCreateNewWaiting(acquire);
        if (!ok) {
            acquire.completeExceptionally((Throwable)new UnexpectedResultException("Objects limit exceeded", Status.of((StatusCode)StatusCode.CLIENT_RESOURCE_EXHAUSTED)));
        }
    }

    public void release(T object) {
        if (!this.used.remove(object, object)) {
            if (!logger.isTraceEnabled()) {
                logger.warn("obj {} double release, possible pool leaks!!", object);
            } else {
                RuntimeException stackTrace = new RuntimeException("Double release");
                logger.warn("obj {} double release, possible pool leaks!!", object, (Object)stackTrace);
            }
            return;
        }
        if (!this.tryToCompleteWaiting(object)) {
            if (this.queueSize.get() > this.limits.maxSize) {
                this.queueSize.decrementAndGet();
                this.handler.destroy(object);
                return;
            }
            this.idle.offerFirst(object);
            if (this.stopped) {
                this.clear();
            }
        }
    }

    public void delete(T object) {
        if (!this.used.remove(object, object)) {
            if (!logger.isTraceEnabled()) {
                logger.warn("obj {} double delete, possible pool leaks!!", object);
            } else {
                RuntimeException stackTrace = new RuntimeException("Double delete");
                logger.warn("obj {} double delete, possible pool leaks!!", object, (Object)stackTrace);
            }
            return;
        }
        this.queueSize.decrementAndGet();
        this.handler.destroy(object);
        this.checkNextWaitingAcquire();
    }

    @Override
    public void close() {
        this.stopped = true;
        this.clear();
    }

    public Iterator<T> coldIterator() {
        return new ColdIterator(this.idle.descendingIterator());
    }

    public int getIdleCount() {
        return this.idle.size();
    }

    public int getUsedCount() {
        return this.used.size();
    }

    public int getTotalCount() {
        return this.queueSize.get();
    }

    public int getPendingCount() {
        return this.pendingRequests.size();
    }

    public int getWaitingCount() {
        return this.waitingAcqueireCount.get();
    }

    public int getTotalLimit() {
        return this.limits.maxSize;
    }

    public int getWaitingLimit() {
        return this.limits.waitingsLimit;
    }

    private boolean safeAcquireObject(CompletableFuture<T> acquire, T object) {
        this.used.put(object, object);
        if (this.stopped) {
            acquire.completeExceptionally(new CancellationException("Queue is already closed"));
            if (this.used.remove(object, object)) {
                this.queueSize.decrementAndGet();
                this.handler.destroy(object);
            }
            return true;
        }
        if (!acquire.complete(object)) {
            this.used.remove(object, object);
            return false;
        }
        return true;
    }

    private boolean tryToPollIdle(CompletableFuture<T> acquire) {
        T next = this.idle.pollFirst();
        if (next == null) {
            return false;
        }
        if (!this.safeAcquireObject(acquire, next)) {
            this.idle.offerFirst(next);
            return false;
        }
        return true;
    }

    private boolean tryToCreateNewPending(CompletableFuture<T> acquire) {
        int count = this.queueSize.get();
        while (count < this.limits.maxSize) {
            if (!this.queueSize.compareAndSet(count, count + 1)) {
                count = this.queueSize.get();
                continue;
            }
            CompletableFuture<T> pending = this.handler.create();
            this.pendingRequests.put(pending, pending);
            pending.whenComplete((BiConsumer)new PendingHandler(acquire, pending));
            return true;
        }
        return false;
    }

    private boolean tryToCreateNewWaiting(CompletableFuture<T> acquire) {
        int waitingsCount = this.waitingAcqueireCount.get();
        while (waitingsCount < this.limits.waitingsLimit) {
            if (!this.waitingAcqueireCount.compareAndSet(waitingsCount, waitingsCount + 1)) {
                waitingsCount = this.waitingAcqueireCount.get();
                continue;
            }
            this.waitingAcquires.offer(acquire);
            return true;
        }
        return false;
    }

    private boolean tryToCompleteWaiting(T object) {
        if (this.stopped) {
            return false;
        }
        CompletableFuture<T> next = this.waitingAcquires.poll();
        while (next != null) {
            this.waitingAcqueireCount.decrementAndGet();
            if (this.safeAcquireObject(next, object)) {
                return true;
            }
            next = this.waitingAcquires.poll();
        }
        return false;
    }

    private void checkNextWaitingAcquire() {
        if (this.stopped || this.waitingAcquires.isEmpty()) {
            return;
        }
        CompletableFuture pending = new CompletableFuture();
        if (this.tryToCreateNewPending(pending)) {
            pending.whenComplete((object, th) -> {
                if (th != null) {
                    this.checkNextWaitingAcquire();
                }
                if (object != null && !this.tryToCompleteWaiting(object)) {
                    this.idle.offerFirst(object);
                }
            });
        }
    }

    private void clear() {
        for (CompletableFuture<T> key : this.pendingRequests.keySet()) {
            if (!this.pendingRequests.remove(key, key)) continue;
            this.queueSize.decrementAndGet();
        }
        CompletableFuture<T> waiting = this.waitingAcquires.poll();
        while (waiting != null) {
            waiting.completeExceptionally(new CancellationException("Queue is already closed"));
            waiting = this.waitingAcquires.poll();
        }
        T nextIdle = this.idle.poll();
        while (nextIdle != null) {
            this.queueSize.decrementAndGet();
            this.handler.destroy(nextIdle);
            nextIdle = this.idle.poll();
        }
    }

    private class ColdIterator
    implements Iterator<T> {
        private final Iterator<T> iter;
        private volatile T lastRet;

        ColdIterator(Iterator<T> iter) {
            this.iter = iter;
        }

        @Override
        public boolean hasNext() {
            return this.iter.hasNext();
        }

        @Override
        public void remove() {
            if (this.lastRet == null) {
                return;
            }
            if (WaitingQueue.this.idle.removeLastOccurrence(this.lastRet)) {
                WaitingQueue.this.handler.destroy(this.lastRet);
                this.lastRet = null;
                WaitingQueue.this.queueSize.decrementAndGet();
                WaitingQueue.this.checkNextWaitingAcquire();
            }
        }

        @Override
        public T next() {
            this.lastRet = this.iter.next();
            return this.lastRet;
        }
    }

    private class PendingHandler
    implements BiConsumer<T, Throwable> {
        private final CompletableFuture<T> acquire;
        private final CompletableFuture<T> pending;

        PendingHandler(CompletableFuture<T> acquire, CompletableFuture<T> pending) {
            this.acquire = acquire;
            this.pending = pending;
        }

        @Override
        public void accept(T object, Throwable th) {
            if (!WaitingQueue.this.pendingRequests.remove(this.pending, this.pending)) {
                this.acquire.completeExceptionally(new CancellationException("Queue is already closed"));
                if (object != null) {
                    WaitingQueue.this.handler.destroy(object);
                }
                return;
            }
            if (th != null) {
                WaitingQueue.this.queueSize.decrementAndGet();
                this.acquire.completeExceptionally(th);
                return;
            }
            if (!this.acquire.isDone() && WaitingQueue.this.safeAcquireObject(this.acquire, object)) {
                return;
            }
            WaitingQueue.this.idle.offerFirst(object);
            if (WaitingQueue.this.stopped) {
                WaitingQueue.this.clear();
            }
        }
    }

    private static class Limits {
        private final int maxSize;
        private final int waitingsLimit;

        Limits(int queueSize, int waitingsSize) {
            this.maxSize = queueSize;
            this.waitingsLimit = waitingsSize;
        }
    }

    public static interface Handler<T> {
        public CompletableFuture<T> create();

        public void destroy(T var1);
    }
}

