/*
 * Decompiled with CFR 0.152.
 */
package tech.ydb.query.tools;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.MoreExecutors;
import java.time.Duration;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;
import tech.ydb.core.Result;
import tech.ydb.core.Status;
import tech.ydb.core.StatusCode;
import tech.ydb.core.UnexpectedResultException;
import tech.ydb.core.utils.FutureTools;
import tech.ydb.query.QueryClient;
import tech.ydb.query.QuerySession;

@ParametersAreNonnullByDefault
public class SessionRetryContext {
    private final QueryClient queryClient;
    private final Executor executor;
    private final Duration sessionCreationTimeout;
    private final int maxRetries;
    private final long backoffSlotMillis;
    private final int backoffCeiling;
    private final long fastBackoffSlotMillis;
    private final int fastBackoffCeiling;
    private final boolean retryNotFound;
    private final boolean idempotent;

    private SessionRetryContext(Builder b) {
        this.queryClient = b.queryClient;
        this.executor = b.executor;
        this.sessionCreationTimeout = b.sessionCreationTimeout;
        this.maxRetries = b.maxRetries;
        this.backoffSlotMillis = b.backoffSlotMillis;
        this.backoffCeiling = b.backoffCeiling;
        this.fastBackoffSlotMillis = b.fastBackoffSlotMillis;
        this.fastBackoffCeiling = b.fastBackoffCeiling;
        this.retryNotFound = b.retryNotFound;
        this.idempotent = b.idempotent;
    }

    public static Builder create(QueryClient sessionSupplier) {
        return new Builder(Objects.requireNonNull(sessionSupplier));
    }

    public <T> CompletableFuture<Result<T>> supplyResult(Function<QuerySession, CompletableFuture<Result<T>>> fn) {
        RetryableResultTask<T> task = new RetryableResultTask<T>(fn);
        task.requestSession();
        return task.getFuture();
    }

    public CompletableFuture<Status> supplyStatus(Function<QuerySession, CompletableFuture<Status>> fn) {
        RetryableStatusTask task = new RetryableStatusTask(fn);
        task.requestSession();
        return task.getFuture();
    }

    private boolean canRetry(StatusCode code) {
        return code.isRetryable(this.idempotent) || this.retryNotFound && code == StatusCode.NOT_FOUND;
    }

    private boolean canRetry(Throwable t) {
        Throwable cause = FutureTools.unwrapCompletionException((Throwable)t);
        if (cause instanceof UnexpectedResultException) {
            StatusCode statusCode = ((UnexpectedResultException)cause).getStatus().getCode();
            return this.canRetry(statusCode);
        }
        return false;
    }

    private long backoffTimeMillisInternal(int retryNumber, long backoffSlotMillis, int backoffCeiling) {
        int slots = 1 << Math.min(retryNumber, backoffCeiling);
        long delay = backoffSlotMillis * (long)slots;
        return delay + ThreadLocalRandom.current().nextLong(delay);
    }

    private long slowBackoffTimeMillis(int retryNumber) {
        return this.backoffTimeMillisInternal(retryNumber, this.backoffSlotMillis, this.backoffCeiling);
    }

    private long fastBackoffTimeMillis(int retryNumber) {
        return this.backoffTimeMillisInternal(retryNumber, this.fastBackoffSlotMillis, this.fastBackoffCeiling);
    }

    private long backoffTimeMillis(StatusCode code, int retryNumber) {
        switch (code) {
            case BAD_SESSION: {
                return 0L;
            }
            case ABORTED: 
            case CLIENT_CANCELLED: 
            case CLIENT_INTERNAL_ERROR: 
            case SESSION_BUSY: 
            case TRANSPORT_UNAVAILABLE: 
            case UNAVAILABLE: 
            case UNDETERMINED: {
                return this.fastBackoffTimeMillis(retryNumber);
            }
        }
        return this.slowBackoffTimeMillis(retryNumber);
    }

    private long backoffTimeMillis(Throwable t, int retryNumber) {
        Throwable cause = FutureTools.unwrapCompletionException((Throwable)t);
        if (cause instanceof UnexpectedResultException) {
            StatusCode statusCode = ((UnexpectedResultException)cause).getStatus().getCode();
            return this.backoffTimeMillis(statusCode, retryNumber);
        }
        return this.slowBackoffTimeMillis(retryNumber);
    }

    @ParametersAreNonnullByDefault
    public static final class Builder {
        private final QueryClient queryClient;
        private Executor executor = MoreExecutors.directExecutor();
        private Duration sessionCreationTimeout = Duration.ofSeconds(5L);
        private int maxRetries = 10;
        private long backoffSlotMillis = 500L;
        private int backoffCeiling = 6;
        private long fastBackoffSlotMillis = 5L;
        private int fastBackoffCeiling = 10;
        private boolean retryNotFound = true;
        private boolean idempotent = false;

        public Builder(QueryClient queryClient) {
            this.queryClient = queryClient;
        }

        public Builder executor(Executor executor) {
            this.executor = Objects.requireNonNull(executor);
            return this;
        }

        public Builder sessionCreationTimeout(Duration duration) {
            this.sessionCreationTimeout = duration;
            return this;
        }

        public Builder maxRetries(int maxRetries) {
            this.maxRetries = maxRetries;
            return this;
        }

        public Builder backoffSlot(Duration duration) {
            Preconditions.checkArgument((!duration.isNegative() ? 1 : 0) != 0, (String)"backoffSlot(%s) is negative", (Object)duration);
            this.backoffSlotMillis = duration.toMillis();
            return this;
        }

        public Builder backoffCeiling(int backoffCeiling) {
            this.backoffCeiling = backoffCeiling;
            return this;
        }

        public Builder fastBackoffSlot(Duration duration) {
            Preconditions.checkArgument((!duration.isNegative() ? 1 : 0) != 0, (String)"backoffSlot(%s) is negative", (Object)duration);
            this.fastBackoffSlotMillis = duration.toMillis();
            return this;
        }

        public Builder fastBackoffCeiling(int backoffCeiling) {
            this.fastBackoffCeiling = backoffCeiling;
            return this;
        }

        public Builder retryNotFound(boolean retryNotFound) {
            this.retryNotFound = retryNotFound;
            return this;
        }

        public Builder idempotent(boolean idempotent) {
            this.idempotent = idempotent;
            return this;
        }

        public SessionRetryContext build() {
            return new SessionRetryContext(this);
        }
    }

    private final class RetryableStatusTask
    extends BaseRetryableTask<Status> {
        RetryableStatusTask(Function<QuerySession, CompletableFuture<Status>> fn) {
            super(fn);
        }

        @Override
        StatusCode toStatusCode(Status status) {
            return status.getCode();
        }

        @Override
        Status toFailedResult(Result<QuerySession> sessionResult) {
            return sessionResult.getStatus();
        }
    }

    private final class RetryableResultTask<T>
    extends BaseRetryableTask<Result<T>> {
        RetryableResultTask(Function<QuerySession, CompletableFuture<Result<T>>> fn) {
            super(fn);
        }

        @Override
        StatusCode toStatusCode(Result<T> result) {
            return result.getStatus().getCode();
        }

        @Override
        Result<T> toFailedResult(Result<QuerySession> sessionResult) {
            return sessionResult.map(s -> null);
        }
    }

    private abstract class BaseRetryableTask<R>
    implements Runnable {
        private final CompletableFuture<R> promise = new CompletableFuture();
        private final AtomicInteger retryNumber = new AtomicInteger();
        private final Function<QuerySession, CompletableFuture<R>> fn;

        BaseRetryableTask(Function<QuerySession, CompletableFuture<R>> fn) {
            this.fn = fn;
        }

        CompletableFuture<R> getFuture() {
            return this.promise;
        }

        abstract StatusCode toStatusCode(R var1);

        abstract R toFailedResult(Result<QuerySession> var1);

        @Override
        public void run() {
            if (this.promise.isCancelled()) {
                return;
            }
            SessionRetryContext.this.executor.execute(this::requestSession);
        }

        public void requestSession() {
            CompletableFuture<Result<QuerySession>> sessionFuture = SessionRetryContext.this.queryClient.createSession(SessionRetryContext.this.sessionCreationTimeout);
            if (sessionFuture.isDone() && !sessionFuture.isCompletedExceptionally()) {
                this.acceptSession(sessionFuture.join());
            } else {
                sessionFuture.whenCompleteAsync((result, th) -> {
                    if (result != null) {
                        this.acceptSession((Result<QuerySession>)result);
                    }
                    if (th != null) {
                        this.handleException((Throwable)th);
                    }
                }, SessionRetryContext.this.executor);
            }
        }

        private void acceptSession(@Nonnull Result<QuerySession> sessionResult) {
            if (!sessionResult.isSuccess()) {
                this.handleError(sessionResult.getStatus().getCode(), this.toFailedResult(sessionResult));
                return;
            }
            QuerySession session = (QuerySession)sessionResult.getValue();
            try {
                this.fn.apply(session).whenComplete((fnResult, fnException) -> {
                    try {
                        session.close();
                        if (fnException != null) {
                            this.handleException((Throwable)fnException);
                            return;
                        }
                        StatusCode statusCode = this.toStatusCode(fnResult);
                        if (statusCode == StatusCode.SUCCESS) {
                            this.promise.complete(fnResult);
                        } else {
                            this.handleError(statusCode, fnResult);
                        }
                    }
                    catch (Throwable unexpected) {
                        this.promise.completeExceptionally(unexpected);
                    }
                });
            }
            catch (RuntimeException ex) {
                session.close();
                this.handleException(ex);
            }
        }

        private void scheduleNext(long delayMillis) {
            if (this.promise.isCancelled()) {
                return;
            }
            SessionRetryContext.this.queryClient.getScheduler().schedule(this, delayMillis, TimeUnit.MILLISECONDS);
        }

        private void handleError(@Nonnull StatusCode code, R result) {
            if (!SessionRetryContext.this.canRetry(code)) {
                this.promise.complete(result);
                return;
            }
            int retry = this.retryNumber.incrementAndGet();
            if (retry <= SessionRetryContext.this.maxRetries) {
                long next = SessionRetryContext.this.backoffTimeMillis(code, retry);
                this.scheduleNext(next);
            } else {
                this.promise.complete(result);
            }
        }

        private void handleException(@Nonnull Throwable ex) {
            if (!SessionRetryContext.this.canRetry(ex)) {
                this.promise.completeExceptionally(ex);
                return;
            }
            int retry = this.retryNumber.incrementAndGet();
            if (retry <= SessionRetryContext.this.maxRetries) {
                long next = SessionRetryContext.this.backoffTimeMillis(ex, retry);
                this.scheduleNext(next);
            } else {
                this.promise.completeExceptionally(ex);
            }
        }
    }
}

