/*
 * Decompiled with CFR 0.152.
 */
package tech.ydb.core.impl.discovery;

import java.time.Instant;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tech.ydb.core.Issue;
import tech.ydb.core.Result;
import tech.ydb.core.Status;
import tech.ydb.core.StatusCode;
import tech.ydb.core.UnexpectedResultException;
import tech.ydb.core.impl.discovery.GrpcDiscoveryRpc;
import tech.ydb.core.utils.Async;
import tech.ydb.proto.discovery.DiscoveryProtos;

public class PeriodicDiscoveryTask
implements Runnable {
    private static final Status EMPTY_DISCOVERY = Status.of(StatusCode.CLIENT_DISCOVERY_FAILED).withIssues(Issue.of("Discovery return empty list of endpoints", Issue.Severity.ERROR));
    private static final long DISCOVERY_PERIOD_NORMAL_SECONDS = 60L;
    private static final long DISCOVERY_PERIOD_MIN_SECONDS = 5L;
    private static final Logger logger = LoggerFactory.getLogger(PeriodicDiscoveryTask.class);
    private final ScheduledExecutorService scheduler;
    private final GrpcDiscoveryRpc discoveryRpc;
    private final DiscoveryHandler discoveryHandler;
    private final long waitingTimeoutMillis;
    private final AtomicBoolean updateInProgress = new AtomicBoolean();
    private final State state = new State();
    private volatile ScheduledFuture<?> currentSchedule = null;

    public PeriodicDiscoveryTask(ScheduledExecutorService scheduler, GrpcDiscoveryRpc rpc, DiscoveryHandler handler, long waitingTimeoutMillis) {
        this.scheduler = scheduler;
        this.discoveryRpc = rpc;
        this.discoveryHandler = handler;
        this.waitingTimeoutMillis = waitingTimeoutMillis;
    }

    public void stop() {
        logger.debug("stopping PeriodicDiscoveryTask");
        this.state.stopped = true;
        if (this.currentSchedule != null) {
            this.currentSchedule.cancel(false);
            this.currentSchedule = null;
        }
    }

    public void start() {
        logger.info("Waiting for init discovery...");
        this.runDiscovery();
        this.state.waitReady(this.waitingTimeoutMillis);
        logger.info("Discovery is finished");
    }

    public void startAsync(Runnable readyWatcher) {
        this.scheduler.execute(() -> {
            logger.info("Waiting for init discovery...");
            this.runDiscovery();
            this.state.waitReady(this.waitingTimeoutMillis);
            logger.info("Discovery is finished");
            if (readyWatcher != null) {
                readyWatcher.run();
            }
        });
    }

    @Override
    public void run() {
        if (this.state.stopped) {
            return;
        }
        if (this.discoveryHandler.useMinDiscoveryPeriod()) {
            this.runDiscovery();
        } else if (Instant.now().isAfter(this.state.lastUpdateTime.plusSeconds(60L))) {
            logger.debug("launching discovery in normal mode");
            this.runDiscovery();
        } else {
            logger.trace("no need to run discovery yet");
            this.scheduleNextDiscovery();
        }
    }

    private void scheduleNextDiscovery() {
        logger.debug("schedule next discovery in {} seconds", (Object)5L);
        this.currentSchedule = this.scheduler.schedule(this, 5L, TimeUnit.SECONDS);
    }

    private void handleDiscoveryResponse(Result<DiscoveryProtos.ListEndpointsResult> response) {
        try {
            DiscoveryProtos.ListEndpointsResult result = response.getValue();
            if (result.getEndpointsList().isEmpty()) {
                logger.error("discovery return empty list of endpoints");
                this.state.handleProblem(new UnexpectedResultException("discovery fail", EMPTY_DISCOVERY));
                return;
            }
            logger.debug("successfully received ListEndpoints result with {} endpoints", (Object)result.getEndpointsList().size());
            this.discoveryHandler.handleDiscoveryResult(result);
            this.state.handleOK();
        }
        catch (UnexpectedResultException ex) {
            logger.error("discovery fail {}", response);
            this.state.handleProblem(ex);
        }
    }

    private void runDiscovery() {
        if (!this.updateInProgress.compareAndSet(false, true)) {
            logger.debug("couldn't start update: already in progress");
            return;
        }
        logger.debug("updating endpoints, calling ListEndpoints...");
        this.discoveryRpc.listEndpoints().whenComplete((response, ex) -> {
            if (this.state.stopped) {
                this.updateInProgress.set(false);
                return;
            }
            if (ex != null) {
                Throwable cause = Async.unwrapCompletionException(ex);
                logger.warn("couldn't perform discovery with exception", cause);
                this.state.handleProblem(cause);
            }
            if (response != null) {
                this.handleDiscoveryResponse((Result<DiscoveryProtos.ListEndpointsResult>)response);
            }
            this.updateInProgress.set(false);
            if (this.state.isReady && !this.state.stopped) {
                this.scheduleNextDiscovery();
            }
        });
    }

    private static class State {
        private volatile Instant lastUpdateTime = Instant.now();
        private volatile boolean isReady = false;
        private volatile boolean stopped = false;
        private volatile RuntimeException lastProblem = null;
        private final Object readyLock = new Object();

        private State() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void handleOK() {
            this.lastUpdateTime = Instant.now();
            if (!this.isReady) {
                Object object = this.readyLock;
                synchronized (object) {
                    this.isReady = true;
                    this.lastProblem = null;
                    this.readyLock.notifyAll();
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void handleProblem(Throwable ex) {
            if (this.isReady) {
                logger.error("discovery problem", ex);
                return;
            }
            Object object = this.readyLock;
            synchronized (object) {
                if (this.isReady) {
                    logger.error("discovery problem", ex);
                    return;
                }
                this.isReady = false;
                this.lastProblem = ex instanceof RuntimeException ? (RuntimeException)ex : new RuntimeException("Check ready problem", ex);
                this.readyLock.notifyAll();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void waitReady(long timeoutMillis) {
            if (this.isReady) {
                return;
            }
            Object object = this.readyLock;
            synchronized (object) {
                if (this.isReady) {
                    return;
                }
                if (this.lastProblem != null) {
                    throw this.lastProblem;
                }
                try {
                    this.readyLock.wait(timeoutMillis);
                    if (this.lastProblem != null) {
                        throw this.lastProblem;
                    }
                    if (!this.isReady) {
                        throw new RuntimeException("Ydb transport in not ready");
                    }
                }
                catch (InterruptedException ex) {
                    logger.warn("ydb transport wait for ready interrupted", (Throwable)ex);
                    Thread.currentThread().interrupt();
                }
            }
        }
    }

    public static interface DiscoveryHandler {
        public boolean useMinDiscoveryPeriod();

        public void handleDiscoveryResult(DiscoveryProtos.ListEndpointsResult var1);
    }
}

