/*
 * Decompiled with CFR 0.152.
 */
package tech.ydb.core.impl.pool;

import com.google.common.annotations.VisibleForTesting;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tech.ydb.core.grpc.BalancingSettings;
import tech.ydb.core.impl.pool.EndpointPriorityFactory;
import tech.ydb.core.impl.pool.EndpointRecord;
import tech.ydb.proto.discovery.DiscoveryProtos;

public final class EndpointPool {
    private static final Logger logger = LoggerFactory.getLogger(EndpointPool.class);
    private static final long DISCOVERY_PESSIMIZATION_THRESHOLD = 50L;
    private final BalancingSettings balancingSettings;
    private final ReadWriteLock recordsLock = new ReentrantReadWriteLock();
    private final AtomicInteger pessimizationRatio = new AtomicInteger();
    private List<PriorityEndpoint> records = new ArrayList<PriorityEndpoint>();
    private Map<Integer, PriorityEndpoint> endpointsByNodeId = new HashMap<Integer, PriorityEndpoint>();
    private int bestEndpointsCount = -1;

    public EndpointPool(BalancingSettings balancingSettings) {
        logger.debug("Creating endpoint pool with balancing settings policy: {}", (Object)balancingSettings.getPolicy());
        this.balancingSettings = balancingSettings;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public EndpointRecord getEndpoint(@Nullable Integer preferredNodeID) {
        this.recordsLock.readLock().lock();
        try {
            PriorityEndpoint knownEndpoint;
            if (preferredNodeID != null && (knownEndpoint = this.endpointsByNodeId.get(preferredNodeID)) != null) {
                PriorityEndpoint priorityEndpoint = knownEndpoint;
                return priorityEndpoint;
            }
            if (this.bestEndpointsCount == -1) {
                knownEndpoint = null;
                return knownEndpoint;
            }
            int idx = ThreadLocalRandom.current().nextInt(this.bestEndpointsCount);
            EndpointRecord endpointRecord = this.records.get(idx);
            return endpointRecord;
        }
        finally {
            this.recordsLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<EndpointRecord> setNewState(DiscoveryProtos.ListEndpointsResult result) {
        EndpointPriorityFactory priorityFactory = new EndpointPriorityFactory(this.balancingSettings, result);
        HashSet<String> newKnownEndpoints = new HashSet<String>();
        HashMap<Integer, PriorityEndpoint> newKnownEndpointsByNodeId = new HashMap<Integer, PriorityEndpoint>();
        ArrayList<PriorityEndpoint> newRecords = new ArrayList<PriorityEndpoint>();
        logger.debug("init new state with {} endpoints", (Object)result.getEndpointsCount());
        for (DiscoveryProtos.EndpointInfo info : result.getEndpointsList()) {
            PriorityEndpoint entry = priorityFactory.createEndpoint(info);
            String endpoint = entry.getHostAndPort();
            if (!newKnownEndpoints.contains(endpoint)) {
                logger.debug("added endpoint {}", (Object)entry);
                newKnownEndpoints.add(endpoint);
                if (entry.getNodeId() != 0) {
                    newKnownEndpointsByNodeId.put(entry.getNodeId(), entry);
                }
                newRecords.add(entry);
                continue;
            }
            logger.warn("duplicate endpoint {}", (Object)entry.getHostAndPort());
        }
        newRecords.sort(PriorityEndpoint.COMPARATOR);
        int newBestEndpointsCount = EndpointPool.getBestEndpointsCount(newRecords);
        ArrayList<EndpointRecord> removed = new ArrayList<EndpointRecord>();
        for (PriorityEndpoint entry : this.records) {
            if (newKnownEndpoints.contains(entry.getHostAndPort())) continue;
            removed.add(entry);
        }
        this.recordsLock.writeLock().lock();
        try {
            this.records = newRecords;
            this.endpointsByNodeId = newKnownEndpointsByNodeId;
            this.bestEndpointsCount = newBestEndpointsCount;
            this.pessimizationRatio.set(0);
        }
        finally {
            this.recordsLock.writeLock().unlock();
        }
        return removed;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void pessimizeEndpoint(EndpointRecord endpoint) {
        if (!(endpoint instanceof PriorityEndpoint)) {
            logger.trace("Endpoint {} is unknown", (Object)endpoint);
            return;
        }
        PriorityEndpoint knownEndpoint = (PriorityEndpoint)endpoint;
        if (knownEndpoint.isPessimized()) {
            logger.trace("Endpoint {} is already pessimized", (Object)endpoint);
            return;
        }
        this.recordsLock.writeLock().lock();
        try {
            knownEndpoint.pessimize();
            int newRatio = (this.pessimizationRatio.get() * this.records.size() + 100) / this.records.size();
            this.pessimizationRatio.set(newRatio);
            if (this.needToRunDiscovery()) {
                logger.debug("launching discovery due to pessimization threshold is exceeded: {} is more than {}", (Object)newRatio, (Object)50L);
            }
            this.records.sort(PriorityEndpoint.COMPARATOR);
            this.bestEndpointsCount = EndpointPool.getBestEndpointsCount(this.records);
            logger.info("Endpoint {} was pessimized. New pessimization ratio: {}", (Object)endpoint, (Object)newRatio);
        }
        finally {
            this.recordsLock.writeLock().unlock();
        }
    }

    public boolean needToRunDiscovery() {
        return (long)this.pessimizationRatio.get() > 50L;
    }

    private static int getBestEndpointsCount(List<PriorityEndpoint> records) {
        int pos;
        if (records.isEmpty()) {
            return -1;
        }
        long bestPriority = records.get(0).priority;
        for (pos = 1; pos < records.size() && records.get(pos).priority == bestPriority; ++pos) {
        }
        return pos;
    }

    @VisibleForTesting
    Map<Integer, PriorityEndpoint> getEndpointsByNodeId() {
        return this.endpointsByNodeId;
    }

    @VisibleForTesting
    List<PriorityEndpoint> getRecords() {
        return this.records;
    }

    @VisibleForTesting
    int getBestEndpointCount() {
        return this.bestEndpointsCount;
    }

    @VisibleForTesting
    static class PriorityEndpoint
    extends EndpointRecord {
        static final Comparator<PriorityEndpoint> COMPARATOR = Comparator.comparingLong(PriorityEndpoint::getPriority).thenComparing(EndpointRecord::getHost).thenComparing(EndpointRecord::getPort);
        private long priority;

        PriorityEndpoint(DiscoveryProtos.EndpointInfo endpoint, long priority) {
            super(endpoint.getAddress(), endpoint.getPort(), endpoint.getNodeId());
            this.priority = priority;
        }

        public long getPriority() {
            return this.priority;
        }

        public void pessimize() {
            this.priority = Long.MAX_VALUE;
        }

        public boolean isPessimized() {
            return this.priority == Long.MAX_VALUE;
        }

        @Override
        public String toString() {
            return "PriorityEndpoint{host=" + this.getHost() + ", port=" + this.getPort() + ", node=" + this.getNodeId() + ", priority= " + this.priority + "}";
        }
    }
}

