/*
 * Decompiled with CFR 0.152.
 */
package tech.ydb.coordination.impl;

import com.google.protobuf.ByteString;
import java.time.Clock;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tech.ydb.common.retry.RetryPolicy;
import tech.ydb.coordination.CoordinationSession;
import tech.ydb.coordination.SemaphoreLease;
import tech.ydb.coordination.description.SemaphoreDescription;
import tech.ydb.coordination.description.SemaphoreWatcher;
import tech.ydb.coordination.impl.LeaseImpl;
import tech.ydb.coordination.impl.Rpc;
import tech.ydb.coordination.impl.SessionState;
import tech.ydb.coordination.impl.Stream;
import tech.ydb.coordination.impl.StreamMsg;
import tech.ydb.coordination.settings.CoordinationSessionSettings;
import tech.ydb.coordination.settings.DescribeSemaphoreMode;
import tech.ydb.coordination.settings.WatchSemaphoreMode;
import tech.ydb.core.Issue;
import tech.ydb.core.Result;
import tech.ydb.core.Status;
import tech.ydb.core.StatusCode;

class SessionImpl
implements CoordinationSession {
    private static final Logger logger = LoggerFactory.getLogger(CoordinationSession.class);
    private final Rpc rpc;
    private final Clock clock;
    private final Executor executor;
    private final RetryPolicy retryPolicy;
    private final String nodePath;
    private final Duration connectTimeout;
    private final ByteString protectionKey;
    private final Map<Consumer<CoordinationSession.State>, Consumer<CoordinationSession.State>> listeners = new ConcurrentHashMap<Consumer<CoordinationSession.State>, Consumer<CoordinationSession.State>>();
    private final AtomicReference<SessionState> state = new AtomicReference<SessionState>(SessionState.unstarted());

    SessionImpl(Rpc rpc, Clock clock, String nodePath, CoordinationSessionSettings settings) {
        this.rpc = rpc;
        this.clock = clock;
        this.executor = settings.getExecutor() != null ? settings.getExecutor() : ForkJoinPool.commonPool();
        this.retryPolicy = settings.getRetryPolicy();
        this.nodePath = nodePath;
        this.connectTimeout = settings.getConnectTimeout();
        this.protectionKey = SessionImpl.createRandomKey();
    }

    @Override
    public long getId() {
        return this.state.get().getSessionId();
    }

    @Override
    public CoordinationSession.State getState() {
        return this.state.get().getState();
    }

    public String toString() {
        return this.state.get().toString();
    }

    @Override
    public void addStateListener(Consumer<CoordinationSession.State> listener) {
        if (listener != null) {
            this.listeners.put(listener, listener);
        }
    }

    @Override
    public void removeStateListener(Consumer<CoordinationSession.State> listener) {
        this.listeners.remove(listener);
    }

    @Override
    public CompletableFuture<Status> stop() {
        logger.debug("{} stopped", (Object)this);
        SessionState local = this.state.get();
        while (!this.updateState(local, SessionState.closed())) {
            local = this.state.get();
        }
        return local.stop();
    }

    @Override
    public CompletableFuture<Status> connect() {
        Stream stream;
        SessionState local = this.state.get();
        if (!this.updateState(local, this.makeConnectionState(local, stream = new Stream(this.rpc)))) {
            logger.warn("{} cannot be connected with state {}", (Object)this, (Object)local.getState());
            return CompletableFuture.completedFuture(Status.of((StatusCode)StatusCode.BAD_REQUEST));
        }
        return this.connectToSession(stream, 0L).thenApplyAsync(res -> this.establishNewSession((Result<Long>)res, stream, Collections.emptyList()), this.executor);
    }

    private CompletableFuture<Result<Long>> connectToSession(Stream stream, long sessionID) {
        stream.startStream().whenCompleteAsync((status, th) -> {
            boolean recoverableState;
            if (th != null) {
                logger.warn("{} stream finished with exception", (Object)this, th);
            }
            if (status != null) {
                if (status.isSuccess()) {
                    logger.debug("{} stream finished with status {}", (Object)this, status);
                } else {
                    logger.warn("{} stream finished with status {}", (Object)this, status);
                }
            }
            ArrayList messagesToRetry = new ArrayList();
            for (StreamMsg<?> msg : stream.getMessages()) {
                if (msg.isIdempotent()) {
                    messagesToRetry.add(msg);
                    continue;
                }
                SessionImpl.completeMessageWithBadSession(msg);
            }
            SessionState local = this.state.get();
            boolean bl = recoverableState = local.getState() == CoordinationSession.State.CONNECTED || local.getState() == CoordinationSession.State.RECONNECTING;
            if (recoverableState && local.hasStream(stream)) {
                logger.debug("stream {} starts to recover");
                long disconnectedAt = this.clock.millis();
                this.restoreSession(disconnectedAt, 0, messagesToRetry);
            } else {
                SessionImpl.completeMessagesWithBadSession(messagesToRetry);
                this.updateState(local, this.makeLostState(local));
            }
        }, this.executor);
        return stream.sendSessionStart(sessionID, this.nodePath, this.connectTimeout, this.protectionKey);
    }

    private void reconnect(Stream stream, long disconnectedAt, int retryNum, List<StreamMsg<?>> messagesToRetry) {
        SessionState local = this.state.get();
        if (local.getState() != CoordinationSession.State.RECONNECTING || !local.hasStream(stream)) {
            SessionImpl.completeMessagesWithBadSession(messagesToRetry);
            return;
        }
        this.connectToSession(stream, local.getSessionId()).whenCompleteAsync((res, th) -> {
            SessionState localState;
            boolean recoverable;
            if (res != null && res.isSuccess()) {
                this.establishNewSession((Result<Long>)res, stream, messagesToRetry);
                return;
            }
            if (th != null) {
                logger.warn("{} stream retry {} finished with exception", new Object[]{this, retryNum, th});
            }
            if (res != null) {
                logger.debug("{} stream retry {} finished with status {}", new Object[]{this, retryNum, res.getStatus()});
            }
            boolean bl = recoverable = (localState = this.state.get()).getState() == CoordinationSession.State.RECONNECTING;
            if (recoverable && local.hasStream(stream)) {
                this.restoreSession(disconnectedAt, retryNum + 1, messagesToRetry);
            } else {
                SessionImpl.completeMessagesWithBadSession(messagesToRetry);
            }
        }, this.executor);
    }

    private void restoreSession(long disconnectedAt, int retryCount, List<StreamMsg<?>> messagesToRetry) {
        SessionState local = this.state.get();
        if (local.getState() != CoordinationSession.State.CONNECTED && local.getState() != CoordinationSession.State.RECONNECTING) {
            SessionImpl.completeMessagesWithBadSession(messagesToRetry);
            return;
        }
        long elapsedTimeMs = this.clock.millis() - disconnectedAt;
        long retryInMs = this.retryPolicy.nextRetryMs(retryCount, elapsedTimeMs);
        if (retryInMs < 0L) {
            logger.debug("stream {} lost connection by retry policy");
            this.updateState(local, this.makeLostState(local));
            SessionImpl.completeMessagesWithBadSession(messagesToRetry);
            return;
        }
        Stream stream = new Stream(this.rpc);
        if (!this.updateState(local, this.makeConnectionState(local, stream))) {
            logger.warn("{} cannot be reconnected with state {}", (Object)this, (Object)this.state.get().getState());
            SessionImpl.completeMessagesWithBadSession(messagesToRetry);
            return;
        }
        if (retryInMs > 0L) {
            logger.debug("stream {} shedule next retry {} in {} ms", new Object[]{this, retryCount, retryInMs});
            this.rpc.getScheduler().schedule(() -> this.reconnect(stream, disconnectedAt, retryCount, messagesToRetry), retryInMs, TimeUnit.MILLISECONDS);
        } else {
            logger.debug("stream {} immediatelly retry {}", (Object)this, (Object)retryCount);
            this.reconnect(stream, disconnectedAt, retryCount, messagesToRetry);
        }
    }

    private Status establishNewSession(Result<Long> result, Stream stream, List<StreamMsg<?>> messagesToRetry) {
        SessionState connected;
        if (!result.isSuccess()) {
            return result.getStatus();
        }
        SessionState local = this.state.get();
        if (!this.updateState(local, connected = this.makeConnectedState(local, (Long)result.getValue(), stream))) {
            stream.stop();
            return Status.of((StatusCode)StatusCode.CANCELLED, (Issue[])new Issue[]{Issue.of((String)"{} cannot handle successful session", (Issue.Severity)Issue.Severity.ERROR)});
        }
        for (StreamMsg<?> msg : messagesToRetry) {
            connected.sendMessage(msg);
        }
        return Status.SUCCESS;
    }

    private SessionState makeConnectionState(SessionState local, Stream stream) {
        if (local.getState() == CoordinationSession.State.INITIAL) {
            return SessionState.connecting(stream);
        }
        if (local.getState() == CoordinationSession.State.LOST) {
            return SessionState.reconnecting(stream);
        }
        if (local.getState() == CoordinationSession.State.CONNECTED || local.getState() == CoordinationSession.State.RECONNECTING) {
            return SessionState.disconnected(local, stream);
        }
        return null;
    }

    private SessionState makeConnectedState(SessionState local, long id, Stream stream) {
        if (local.getState() == CoordinationSession.State.CONNECTING && local.hasStream(stream)) {
            return SessionState.connected(local, id);
        }
        if (local.getState() == CoordinationSession.State.RECONNECTING && local.hasStream(stream)) {
            return SessionState.reconnected(local);
        }
        return null;
    }

    private SessionState makeLostState(SessionState local) {
        if (local.getState() == CoordinationSession.State.CONNECTING) {
            return SessionState.unstarted();
        }
        if (local.getState() == CoordinationSession.State.RECONNECTING) {
            return SessionState.lost();
        }
        return null;
    }

    private boolean updateState(SessionState previous, SessionState next) {
        if (next == null || !this.state.compareAndSet(previous, next)) {
            return false;
        }
        if (next.getState() != previous.getState()) {
            for (Consumer<CoordinationSession.State> listener : this.listeners.values()) {
                listener.accept(next.getState());
            }
        }
        return true;
    }

    @Override
    public CompletableFuture<Status> createSemaphore(String name, long limit, byte[] data) {
        StreamMsg<Status> msg = StreamMsg.createSemaphore(name, limit, data);
        this.state.get().sendMessage(msg);
        return msg.getResult().thenApplyAsync(Function.identity(), this.executor);
    }

    @Override
    public CompletableFuture<Status> updateSemaphore(String name, byte[] data) {
        StreamMsg<Status> msg = StreamMsg.updateSemaphore(name, data);
        this.state.get().sendMessage(msg);
        return msg.getResult().thenApplyAsync(Function.identity(), this.executor);
    }

    @Override
    public CompletableFuture<Status> deleteSemaphore(String name, boolean force) {
        StreamMsg<Status> msg = StreamMsg.deleteSemaphore(name, force);
        this.state.get().sendMessage(msg);
        return msg.getResult().thenApplyAsync(Function.identity(), this.executor);
    }

    @Override
    public CompletableFuture<Result<SemaphoreDescription>> describeSemaphore(String name, DescribeSemaphoreMode mode) {
        StreamMsg<Result<SemaphoreDescription>> msg = StreamMsg.describeSemaphore(name, mode);
        this.state.get().sendMessage(msg);
        return msg.getResult().thenApplyAsync(Function.identity(), this.executor);
    }

    @Override
    public CompletableFuture<Result<SemaphoreWatcher>> watchSemaphore(String name, DescribeSemaphoreMode describeMode, WatchSemaphoreMode watchMode) {
        StreamMsg<Result<SemaphoreWatcher>> msg = StreamMsg.watchSemaphore(name, describeMode, watchMode);
        this.state.get().sendMessage(msg);
        return msg.getResult().thenApplyAsync(Function.identity(), this.executor);
    }

    @Override
    public CompletableFuture<Result<SemaphoreLease>> acquireSemaphore(String name, long count, byte[] data, Duration timeout) {
        StreamMsg<Result<Boolean>> msg = StreamMsg.acquireSemaphore(name, count, data, false, timeout.toMillis());
        this.state.get().sendMessage(msg);
        return msg.getResult().thenApplyAsync((Function)new LeaseCreator(name), this.executor);
    }

    @Override
    public CompletableFuture<Result<SemaphoreLease>> acquireEphemeralSemaphore(String name, boolean exclusive, byte[] data, Duration timeout) {
        long count = exclusive ? -1L : 1L;
        StreamMsg<Result<Boolean>> msg = StreamMsg.acquireSemaphore(name, count, data, true, timeout.toMillis());
        this.state.get().sendMessage(msg);
        return msg.getResult().thenApplyAsync((Function)new LeaseCreator(name), this.executor);
    }

    CompletableFuture<Boolean> releaseSemaphore(String name) {
        StreamMsg<Result<Boolean>> msg = StreamMsg.releaseSemaphore(name);
        this.state.get().sendMessage(msg);
        return msg.getResult().thenApplyAsync(result -> result.isSuccess() && (Boolean)result.getValue() != false, this.executor);
    }

    private static void completeMessageWithBadSession(StreamMsg<?> msg) {
        for (StreamMsg<?> local = msg; local != null; local = local.nextMsg()) {
            local.handleError(Status.of((StatusCode)StatusCode.BAD_SESSION));
        }
    }

    private static void completeMessagesWithBadSession(Collection<StreamMsg<?>> messages) {
        for (StreamMsg<?> msg : messages) {
            SessionImpl.completeMessageWithBadSession(msg);
        }
    }

    private static ByteString createRandomKey() {
        byte[] protectionKey = new byte[16];
        ThreadLocalRandom.current().nextBytes(protectionKey);
        return ByteString.copyFrom((byte[])protectionKey);
    }

    private class LeaseCreator
    implements Function<Result<Boolean>, Result<SemaphoreLease>> {
        private final String name;

        LeaseCreator(String name) {
            this.name = name;
        }

        @Override
        public Result<SemaphoreLease> apply(Result<Boolean> acquireResult) {
            if (!acquireResult.isSuccess()) {
                return acquireResult.map(null);
            }
            if (!((Boolean)acquireResult.getValue()).booleanValue()) {
                return Result.fail((Status)Status.of((StatusCode)StatusCode.TIMEOUT));
            }
            return Result.success((Object)new LeaseImpl(SessionImpl.this, this.name));
        }
    }
}

