/*
 * Decompiled with CFR 0.152.
 */
package tech.ydb.core.auth;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.EnumSet;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;
import tech.ydb.auth.AuthIdentity;
import tech.ydb.auth.AuthRpcProvider;
import tech.ydb.core.Result;
import tech.ydb.core.StatusCode;
import tech.ydb.core.UnexpectedResultException;
import tech.ydb.core.auth.BackgroundIdentity;
import tech.ydb.core.auth.JwtUtils;
import tech.ydb.core.grpc.GrpcRequestSettings;
import tech.ydb.core.grpc.GrpcTransport;
import tech.ydb.core.impl.auth.GrpcAuthRpc;
import tech.ydb.core.operation.OperationBinder;
import tech.ydb.proto.auth.YdbAuth;
import tech.ydb.proto.auth.v1.AuthServiceGrpc;
import tech.ydb.shaded.google.common.annotations.VisibleForTesting;
import tech.ydb.shaded.slf4j.Logger;
import tech.ydb.shaded.slf4j.LoggerFactory;

public class StaticCredentials
implements AuthRpcProvider<GrpcAuthRpc> {
    private static final Logger logger = LoggerFactory.getLogger(StaticCredentials.class);
    private static final int LOGIN_TIMEOUT_SECONDS = 10;
    private static final int MAX_RETRIES_COUNT = 5;
    private static final EnumSet<StatusCode> RETRYABLE_STATUSES = EnumSet.of(StatusCode.ABORTED, new StatusCode[]{StatusCode.UNAVAILABLE, StatusCode.OVERLOADED, StatusCode.CLIENT_RESOURCE_EXHAUSTED, StatusCode.BAD_SESSION, StatusCode.SESSION_BUSY, StatusCode.UNDETERMINED, StatusCode.TRANSPORT_UNAVAILABLE});
    private final Clock clock;
    private final YdbAuth.LoginRequest request;

    @VisibleForTesting
    StaticCredentials(Clock clock, String username, String password) {
        this.clock = clock;
        YdbAuth.LoginRequest.Builder builder = YdbAuth.LoginRequest.newBuilder().setUser(username);
        if (password != null) {
            builder.setPassword(password);
        }
        this.request = builder.build();
    }

    public StaticCredentials(String username, String password) {
        this(Clock.systemUTC(), username, password);
    }

    @Override
    public AuthIdentity createAuthIdentity(GrpcAuthRpc rpc) {
        logger.info("create static identity for database {}", (Object)rpc.getDatabase());
        return new BackgroundIdentity(this.clock, new LoginRpc(rpc));
    }

    private class LoginRpc
    implements BackgroundIdentity.Rpc {
        private final GrpcAuthRpc rpc;
        private final AtomicInteger retries = new AtomicInteger(5);

        LoginRpc(GrpcAuthRpc rpc) {
            this.rpc = rpc;
        }

        private void handleResult(CompletableFuture<BackgroundIdentity.Rpc.Token> future, Result<YdbAuth.LoginResult> resp) {
            if (resp.isSuccess()) {
                try {
                    Instant now = StaticCredentials.this.clock.instant();
                    String token = resp.getValue().getToken();
                    Instant expiredAt = JwtUtils.extractExpireAt(token, now);
                    long expiresIn = expiredAt.getEpochSecond() - now.getEpochSecond();
                    Instant updateAt = now.plus(expiresIn / 2L, ChronoUnit.SECONDS);
                    updateAt = updateAt.isBefore(now) ? now : updateAt;
                    logger.debug("logged in with expired at {} and updating at {}", (Object)expiredAt, (Object)updateAt);
                    future.complete(new BackgroundIdentity.Rpc.Token(token, expiredAt, updateAt));
                }
                catch (RuntimeException ex) {
                    future.completeExceptionally(ex);
                }
            } else {
                logger.error("Login request get wrong status {}", (Object)resp.getStatus());
                if (RETRYABLE_STATUSES.contains((Object)resp.getStatus().getCode()) && this.retries.decrementAndGet() > 0) {
                    this.tryLogin(future);
                } else {
                    future.completeExceptionally(new UnexpectedResultException("Can't login", resp.getStatus()));
                }
            }
        }

        private void handleException(CompletableFuture<BackgroundIdentity.Rpc.Token> future, Throwable th) {
            logger.error("Login request get exception {}", (Object)th.getMessage());
            if (this.retries.decrementAndGet() > 0) {
                this.tryLogin(future);
            } else {
                future.completeExceptionally(th);
            }
        }

        private void tryLogin(CompletableFuture<BackgroundIdentity.Rpc.Token> future) {
            if (future.isCancelled() || future.isDone()) {
                return;
            }
            this.rpc.getExecutor().submit(() -> {
                try (GrpcTransport transport = this.rpc.createTransport();){
                    GrpcRequestSettings grpcSettings = GrpcRequestSettings.newBuilder().withDeadline(Duration.ofSeconds(10L)).build();
                    ((CompletableFuture)((CompletableFuture)transport.unaryCall(AuthServiceGrpc.getLoginMethod(), grpcSettings, StaticCredentials.this.request).thenApply(OperationBinder.bindSync(YdbAuth.LoginResponse::getOperation, YdbAuth.LoginResult.class))).whenComplete((resp, th) -> {
                        if (resp != null) {
                            this.handleResult(future, (Result<YdbAuth.LoginResult>)resp);
                        }
                        if (th != null) {
                            this.handleException(future, (Throwable)th);
                        }
                    })).join();
                }
            });
        }

        @Override
        public CompletableFuture<BackgroundIdentity.Rpc.Token> getTokenAsync() {
            CompletableFuture<BackgroundIdentity.Rpc.Token> tokenFuture = new CompletableFuture<BackgroundIdentity.Rpc.Token>();
            tokenFuture.whenComplete((token, th) -> {
                if (token == null || th != null) {
                    this.rpc.changeEndpoint();
                }
            });
            this.tryLogin(tokenFuture);
            return tokenFuture;
        }

        @Override
        public int getTimeoutSeconds() {
            return 10;
        }
    }
}

