/*
 * Decompiled with CFR 0.152.
 */
package tech.ydb.jdbc.context;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import java.sql.SQLDataException;
import java.sql.SQLException;
import java.time.Duration;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import tech.ydb.core.Result;
import tech.ydb.core.UnexpectedResultException;
import tech.ydb.core.grpc.GrpcTransport;
import tech.ydb.core.grpc.GrpcTransportBuilder;
import tech.ydb.jdbc.YdbPrepareMode;
import tech.ydb.jdbc.context.YdbConfig;
import tech.ydb.jdbc.exception.ExceptionFactory;
import tech.ydb.jdbc.query.JdbcParams;
import tech.ydb.jdbc.query.JdbcQueryLexer;
import tech.ydb.jdbc.query.YdbQuery;
import tech.ydb.jdbc.query.YdbQueryBuilder;
import tech.ydb.jdbc.query.YdbQueryOptions;
import tech.ydb.jdbc.query.params.BatchedParams;
import tech.ydb.jdbc.query.params.InMemoryParams;
import tech.ydb.jdbc.query.params.PreparedParams;
import tech.ydb.jdbc.settings.ParsedProperty;
import tech.ydb.jdbc.settings.YdbClientProperties;
import tech.ydb.jdbc.settings.YdbClientProperty;
import tech.ydb.jdbc.settings.YdbConnectionProperties;
import tech.ydb.jdbc.settings.YdbConnectionProperty;
import tech.ydb.jdbc.settings.YdbOperationProperties;
import tech.ydb.scheme.SchemeClient;
import tech.ydb.table.SessionRetryContext;
import tech.ydb.table.TableClient;
import tech.ydb.table.description.TableDescription;
import tech.ydb.table.impl.PooledTableClient;
import tech.ydb.table.query.DataQuery;
import tech.ydb.table.rpc.grpc.GrpcTableRpc;
import tech.ydb.table.settings.DescribeTableSettings;
import tech.ydb.table.settings.PrepareDataQuerySettings;
import tech.ydb.table.settings.RequestSettings;
import tech.ydb.table.values.Type;

public class YdbContext
implements AutoCloseable {
    private static final Logger LOGGER = Logger.getLogger(YdbContext.class.getName());
    private static final int SESSION_POOL_DEFAULT_MIN_SIZE = 0;
    private static final int SESSION_POOL_DEFAULT_MAX_SIZE = 50;
    private static final int SESSION_POOL_RESIZE_STEP = 50;
    private static final int SESSION_POOL_RESIZE_THRESHOLD = 10;
    private final YdbConfig config;
    private final GrpcTransport grpcTransport;
    private final PooledTableClient tableClient;
    private final SchemeClient schemeClient;
    private final YdbQueryOptions queryOptions;
    private final SessionRetryContext retryCtx;
    private final Cache<String, YdbQuery> queriesCache;
    private final Cache<String, Map<String, Type>> queryParamsCache;
    private final boolean autoResizeSessionPool;
    private final AtomicInteger connectionsCount = new AtomicInteger();

    private YdbContext(YdbConfig config, GrpcTransport transport, PooledTableClient tableClient, boolean autoResize) {
        this.config = config;
        this.grpcTransport = Objects.requireNonNull(transport);
        this.tableClient = Objects.requireNonNull(tableClient);
        this.schemeClient = SchemeClient.newClient(transport).build();
        this.queryOptions = YdbQueryOptions.createFrom(config.getOperationProperties());
        this.autoResizeSessionPool = autoResize;
        this.retryCtx = SessionRetryContext.create(tableClient).build();
        int cacheSize = config.getOperationProperties().getPreparedStatementCacheSize();
        if (cacheSize > 0) {
            this.queriesCache = CacheBuilder.newBuilder().maximumSize(cacheSize).build();
            this.queryParamsCache = CacheBuilder.newBuilder().maximumSize(cacheSize).build();
        } else {
            this.queriesCache = null;
            this.queryParamsCache = null;
        }
    }

    public String getDatabase() {
        return this.grpcTransport.getDatabase();
    }

    public SchemeClient getSchemeClient() {
        return this.schemeClient;
    }

    public TableClient getTableClient() {
        return this.tableClient;
    }

    public String getUrl() {
        return this.config.getUrl();
    }

    public int getConnectionsCount() {
        return this.connectionsCount.get();
    }

    public YdbOperationProperties getOperationProperties() {
        return this.config.getOperationProperties();
    }

    @Override
    public void close() {
        try {
            this.schemeClient.close();
            this.tableClient.close();
            this.grpcTransport.close();
        }
        catch (Exception e) {
            LOGGER.log(Level.SEVERE, "Unable to close client: " + e.getMessage(), e);
        }
    }

    public void register() {
        int actual = this.connectionsCount.incrementAndGet();
        int maxSize = this.tableClient.sessionPoolStats().getMaxSize();
        if (this.autoResizeSessionPool && actual > maxSize - 10) {
            int newSize = maxSize + 50;
            if (maxSize == this.tableClient.sessionPoolStats().getMaxSize()) {
                this.tableClient.updatePoolMaxSize(newSize);
            }
        }
    }

    public void deregister() {
        int actual = this.connectionsCount.decrementAndGet();
        int maxSize = this.tableClient.sessionPoolStats().getMaxSize();
        if (this.autoResizeSessionPool && maxSize > 50 && actual < maxSize - 50 - 20) {
            int newSize = maxSize - 50;
            if (maxSize == this.tableClient.sessionPoolStats().getMaxSize()) {
                this.tableClient.updatePoolMaxSize(newSize);
            }
        }
    }

    public static YdbContext createContext(YdbConfig config) throws SQLException {
        try {
            YdbConnectionProperties connProps = config.getConnectionProperties();
            YdbClientProperties clientProps = config.getClientProperties();
            LOGGER.log(Level.INFO, "Creating new YDB connection to {0}", connProps.getConnectionString());
            GrpcTransport grpcTransport = YdbContext.buildGrpcTransport(connProps);
            PooledTableClient.Builder tableClient = PooledTableClient.newClient(GrpcTableRpc.useTransport(grpcTransport));
            boolean autoResize = YdbContext.buildTableClient(tableClient, clientProps);
            return new YdbContext(config, grpcTransport, tableClient.build(), autoResize);
        }
        catch (RuntimeException ex) {
            throw new SQLException("Cannot connect to YDB: " + ex.getMessage(), ex);
        }
    }

    public static GrpcTransport buildGrpcTransport(YdbConnectionProperties props) {
        GrpcTransportBuilder builder = GrpcTransport.forConnectionString(props.getConnectionString());
        for (Map.Entry<YdbConnectionProperty<?>, ParsedProperty> entry : props.getParams().entrySet()) {
            if (entry.getValue() == null) continue;
            entry.getKey().getSetter().accept(builder, (GrpcTransportBuilder)entry.getValue().getParsedValue());
        }
        if (props.hasStaticCredentials()) {
            builder = builder.withAuthProvider(props.getStaticCredentials());
        }
        builder.withSchedulerFactory(() -> {
            String namePrefix = "ydb-jdbc-scheduler[" + props.hashCode() + "]-thread-";
            AtomicInteger threadNumber = new AtomicInteger(1);
            return Executors.newScheduledThreadPool(1, r -> {
                Thread t = new Thread(r, namePrefix + threadNumber.getAndIncrement());
                t.setDaemon(true);
                return t;
            });
        });
        return builder.build();
    }

    private static boolean buildTableClient(TableClient.Builder builder, YdbClientProperties props) {
        for (Map.Entry<YdbClientProperty<?>, ParsedProperty> entry : props.getParams().entrySet()) {
            if (entry.getValue() == null) continue;
            entry.getKey().getSetter().accept(builder, (TableClient.Builder)entry.getValue().getParsedValue());
        }
        ParsedProperty minSizeConfig = props.getProperty(YdbClientProperty.SESSION_POOL_SIZE_MIN);
        ParsedProperty maxSizeConfig = props.getProperty(YdbClientProperty.SESSION_POOL_SIZE_MAX);
        if (minSizeConfig == null && maxSizeConfig == null) {
            return true;
        }
        int minSize = 0;
        int maxSize = 50;
        if (minSizeConfig != null) {
            minSize = Math.max(0, (Integer)minSizeConfig.getParsedValue());
            maxSize = Math.max(maxSize, minSize);
        }
        if (maxSizeConfig != null) {
            maxSize = Math.max(minSize + 1, (Integer)maxSizeConfig.getParsedValue());
        }
        builder.sessionPoolSize(minSize, maxSize);
        return false;
    }

    public <T extends RequestSettings<?>> T withDefaultTimeout(T settings) {
        Duration operation = this.config.getOperationProperties().getDeadlineTimeout();
        if (!operation.isZero() && !operation.isNegative()) {
            settings.setOperationTimeout(operation);
            settings.setTimeout(operation.plusSeconds(1L));
        }
        return settings;
    }

    public CompletableFuture<Result<TableDescription>> describeTable(String tablePath, DescribeTableSettings settings) {
        return this.retryCtx.supplyResult(session -> session.describeTable(tablePath, settings));
    }

    public YdbQuery parseYdbQuery(String sql) throws SQLException {
        YdbQueryBuilder builder = new YdbQueryBuilder(sql, this.queryOptions.getForcedQueryType());
        JdbcQueryLexer.buildQuery(builder, this.queryOptions);
        return builder.build(this.queryOptions);
    }

    public YdbQuery findOrParseYdbQuery(String sql) throws SQLException {
        if (this.queriesCache == null) {
            return this.parseYdbQuery(sql);
        }
        YdbQuery cached = this.queriesCache.getIfPresent(sql);
        if (cached == null) {
            cached = this.parseYdbQuery(sql);
            this.queriesCache.put(sql, cached);
        }
        return cached;
    }

    public JdbcParams findOrCreateJdbcParams(YdbQuery query, YdbPrepareMode mode) throws SQLException {
        if (query.hasIndexesParameters() || mode == YdbPrepareMode.IN_MEMORY || !this.queryOptions.iPrepareDataQueries()) {
            return new InMemoryParams(query.getIndexesParameters());
        }
        String yql = query.getYqlQuery(null);
        PrepareDataQuerySettings settings = this.withDefaultTimeout(new PrepareDataQuerySettings());
        try {
            boolean requireBatch;
            Map<String, Type> types = this.queryParamsCache.getIfPresent(query.originSQL());
            if (types == null) {
                types = ((DataQuery)this.retryCtx.supplyResult(session -> session.prepareDataQuery(yql, settings)).join().getValue()).types();
                this.queryParamsCache.put(query.originSQL(), types);
            }
            boolean bl = requireBatch = mode == YdbPrepareMode.DATA_QUERY_BATCH;
            if (requireBatch || mode == YdbPrepareMode.AUTO && this.queryOptions.isDetectBatchQueries()) {
                BatchedParams params = BatchedParams.tryCreateBatched(types);
                if (params != null) {
                    return params;
                }
                if (requireBatch) {
                    throw new SQLDataException("Statement cannot be executed as batch statement: " + query.originSQL());
                }
            }
            return new PreparedParams(types);
        }
        catch (UnexpectedResultException ex) {
            throw ExceptionFactory.createException("Cannot prepare data query: " + ex.getMessage(), ex);
        }
    }
}

