/*
 * Decompiled with CFR 0.152.
 */
package tech.ydb.core.auth;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.EnumSet;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tech.ydb.core.Operations;
import tech.ydb.core.Result;
import tech.ydb.core.StatusCode;
import tech.ydb.core.UnexpectedResultException;
import tech.ydb.core.auth.JwtUtils;
import tech.ydb.core.grpc.GrpcRequestSettings;
import tech.ydb.core.grpc.GrpcTransport;
import tech.ydb.core.impl.auth.GrpcAuthRpc;
import tech.ydb.proto.auth.YdbAuth;
import tech.ydb.proto.auth.v1.AuthServiceGrpc;

class StaticCredentialsRpc {
    private static final Logger logger = LoggerFactory.getLogger(StaticCredentialsRpc.class);
    private static final long LOGIN_TIMEOUT_SECONDS = 10L;
    private static final int MAX_RETRIES_COUNT = 5;
    private static final EnumSet<StatusCode> RETRYABLE_STATUSES = EnumSet.of(StatusCode.ABORTED, new StatusCode[]{StatusCode.UNAVAILABLE, StatusCode.OVERLOADED, StatusCode.CLIENT_RESOURCE_EXHAUSTED, StatusCode.BAD_SESSION, StatusCode.SESSION_BUSY, StatusCode.UNDETERMINED, StatusCode.TRANSPORT_UNAVAILABLE});
    private final AtomicInteger retries = new AtomicInteger(5);
    private final GrpcAuthRpc rpc;
    private final YdbAuth.LoginRequest request;
    private final Clock clock;

    StaticCredentialsRpc(GrpcAuthRpc rpc, YdbAuth.LoginRequest request, Clock clock) {
        this.rpc = rpc;
        this.request = request;
        this.clock = clock;
    }

    private void handleResult(CompletableFuture<Token> future, Result<YdbAuth.LoginResult> resp) {
        if (resp.isSuccess()) {
            try {
                Instant now = this.clock.instant();
                String token = resp.getValue().getToken();
                Instant expiredAt = JwtUtils.extractExpireAt(token, now);
                long expiresIn = expiredAt.getEpochSecond() - now.getEpochSecond();
                Instant updateAt = now.plus(expiresIn / 2L, ChronoUnit.SECONDS);
                updateAt = updateAt.isBefore(now) ? now : updateAt;
                future.complete(new Token(token, expiredAt, updateAt));
            }
            catch (RuntimeException ex) {
                future.completeExceptionally(ex);
            }
        } else {
            logger.error("Login request get wrong status {}", (Object)resp.getStatus());
            if (RETRYABLE_STATUSES.contains((Object)resp.getStatus().getCode()) && this.retries.decrementAndGet() > 0) {
                this.tryLogin(future);
            } else {
                future.completeExceptionally(new UnexpectedResultException("Can't login", resp.getStatus()));
            }
        }
    }

    private void handleException(CompletableFuture<Token> future, Throwable th) {
        logger.error("Login request get exception {}", (Object)th.getMessage());
        if (this.retries.decrementAndGet() > 0) {
            this.tryLogin(future);
        } else {
            future.completeExceptionally(th);
        }
    }

    private void tryLogin(CompletableFuture<Token> future) {
        if (future.isCancelled() || future.isDone()) {
            return;
        }
        this.rpc.getExecutor().submit(() -> {
            try (GrpcTransport transport = this.rpc.createTransport();){
                GrpcRequestSettings grpcSettings = GrpcRequestSettings.newBuilder().withDeadline(Duration.ofSeconds(10L)).build();
                ((CompletableFuture)((CompletableFuture)transport.unaryCall(AuthServiceGrpc.getLoginMethod(), grpcSettings, this.request).thenApply(Operations.resultUnwrapper(YdbAuth.LoginResponse::getOperation, YdbAuth.LoginResult.class))).whenComplete((resp, th) -> {
                    if (resp != null) {
                        this.handleResult(future, (Result<YdbAuth.LoginResult>)resp);
                    }
                    if (th != null) {
                        this.handleException(future, (Throwable)th);
                    }
                })).join();
            }
        });
    }

    public CompletableFuture<Token> loginAsync() {
        CompletableFuture<Token> tokenFuture = new CompletableFuture<Token>();
        tokenFuture.whenComplete((token, th) -> {
            if (token == null || th != null) {
                this.rpc.changeEndpoint();
            }
        });
        this.tryLogin(tokenFuture);
        return tokenFuture;
    }

    public <T> T unwrap(CompletableFuture<T> future) {
        try {
            return future.get(10L, TimeUnit.SECONDS);
        }
        catch (ExecutionException | TimeoutException ex) {
            logger.error("static authentication problem", ex);
            throw new RuntimeException("static authentication problem", ex);
        }
        catch (InterruptedException ex) {
            logger.error("validation of static credentials token is interupted", ex);
            Thread.currentThread().interrupt();
            return null;
        }
    }

    public static class Token {
        private final String token;
        private final Instant expiredAt;
        private final Instant updateAt;

        Token(String token, Instant expiredAt, Instant updateAt) {
            this.token = token;
            this.expiredAt = expiredAt;
            this.updateAt = updateAt;
        }

        public String token() {
            return this.token;
        }

        public Instant expiredAt() {
            return this.expiredAt;
        }

        public Instant updateAt() {
            return this.updateAt;
        }
    }
}

