package tech.xmagic.api.advice;


import cn.hutool.core.util.StrUtil;
import jakarta.validation.ConstraintViolation;
import jakarta.validation.ConstraintViolationException;
import lombok.extern.slf4j.Slf4j;
import tech.xmagic.api.Result;
import tech.xmagic.api.autoconfigure.ApiProperties;

import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.core.annotation.Order;
import org.springframework.http.HttpStatus;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingPathVariableException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.HttpServerErrorException;
import tech.xmagic.enums.RC;
import tech.xmagic.exception.AbstractException;
import tech.xmagic.exception.AbstractRuntimeException;

import java.util.Set;


/**
 * 全局异常处理
 *
 * @since 1.0.0
 */
@Order(0)
@Slf4j
@ConditionalOnProperty(prefix = ApiProperties.PREFIX, name = ApiProperties.HTTP_STATUS_NAME, havingValue = "500")
@RestControllerAdvice
public class ExceptionControllerAdvice500 {
    public ExceptionControllerAdvice500(ApiProperties apiProperties) {
        this.apiProperties = apiProperties;
    }

    //    @Autowired
    private ApiProperties apiProperties;
    /**
     * 处理MissingPathVariableException异常
     *
     * @param e
     * @return
     */
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    @ExceptionHandler(MissingPathVariableException.class)
    public Result<Void> MissingPathVariableExceptionHandler(MissingPathVariableException e) {
        String message = "缺少路径参数:" + e.getVariableName();
        Result<Void> result = Result.failure(RC.VALIDATION_FAILURE, message);
        if (StrUtil.isNotBlank(apiProperties.getSource()))
            result.setSource(apiProperties.getSource());
        log.error(result.toString(), e);
        return result;
    }

    /**
     * 处理MissingServletRequestParameterException异常
     *
     * @param e
     * @return
     */
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    @ExceptionHandler(MissingServletRequestParameterException.class)
    public Result<Void> MissingServletRequestParameterExceptionHandler(MissingServletRequestParameterException e) {
        String message = "缺少参数:" + e.getParameterName();
        Result<Void> result = Result.failure(RC.VALIDATION_FAILURE, message);
        if (StrUtil.isNotBlank(apiProperties.getSource()))
            result.setSource(apiProperties.getSource());
        log.error(result.toString(), e);
        return result;
    }
    /**
     * 处理@Valid注解校验产生的异常
     *
     * @param e
     * @return
     */
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    @ExceptionHandler(MethodArgumentNotValidException.class)
    public Result<Void> MethodArgumentNotValidExceptionHandler(MethodArgumentNotValidException e) {
        BindingResult bindingResult = e.getBindingResult();
        StringBuffer sb = new StringBuffer();
        for (FieldError fieldError : bindingResult.getFieldErrors()) {
//            sb.append(fieldError.getField() + "|" + fieldError.getDefaultMessage() + "/");
            sb.append(fieldError.getField() + "|" + fieldError.getDefaultMessage() + "/");
        }
//        ObjectError objectError = e.getBindingResult().getAllErrors().get(0);
//        String msg = objectError.getDefaultMessage();
        Result<Void> result = Result.failure(RC.VALIDATION_FAILURE, sb.toString());
        if (StrUtil.isNotBlank(apiProperties.getSource()))
            result.setSource(apiProperties.getSource());
        log.error(result.toString(), e);
        return result;
    }

    /**
     * 处理@Valid注解校验产生的异常
     *
     * @param e
     * @return
     */
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    @ExceptionHandler(BindException.class)
    public Result<Void> BindExceptionHandler(BindException e) {
        BindingResult bindingResult = e.getBindingResult();
        StringBuffer sb = new StringBuffer();
        for (FieldError fieldError : bindingResult.getFieldErrors()) {
//            sb.append(fieldError.getField() + "|" + fieldError.getDefaultMessage() + "/");
            sb.append(fieldError.getField() + "|" + fieldError.getDefaultMessage() + "/");
        }
//        ObjectError objectError = e.getBindingResult().getAllErrors().get(0);
//        String msg = objectError.getDefaultMessage();
        Result<Void> result = Result.failure(RC.VALIDATION_FAILURE, sb.toString());
        if (StrUtil.isNotBlank(apiProperties.getSource()))
            result.setSource(apiProperties.getSource());
        log.error(result.toString(), e);
        return result;
    }

    /**
     * 处理@Valid注解校验产生的异常
     *
     * @param e
     * @return
     */
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    @ExceptionHandler(ConstraintViolationException.class)
    public Result<Void> ConstraintViolationExceptionHandler(ConstraintViolationException e) {
        StringBuffer sb = new StringBuffer();
        Set<ConstraintViolation<?>> violations = e.getConstraintViolations();
        for (ConstraintViolation<?> item : violations) {
            sb.append(item.getMessage() + "/");
        }
        Result<Void> result = Result.failure(RC.VALIDATION_FAILURE, sb.toString());
        if (StrUtil.isNotBlank(apiProperties.getSource()))
            result.setSource(apiProperties.getSource());
        log.error(result.toString(), e);
        return result;
    }


    /**
     * 所有HttpClientErrorException异常的处理
     * 8000+
     *
     * @param e
     * @return
     */
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    @ExceptionHandler(HttpClientErrorException.class)
    public Result<Void> httpClientErrorExceptionHandler(HttpClientErrorException e) {
        Result<Void> result = Result.error(e);
        if (StrUtil.isNotBlank(apiProperties.getSource()))
            result.setSource(apiProperties.getSource());
        log.error(result.toString(), e);
        return result;
    }

    /**
     * 所有HttpServerErrorException异常的处理
     * 8000+
     *
     * @param e
     * @return
     */
    @ExceptionHandler(HttpServerErrorException.class)
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public Result<Void> httpServerErrorExceptionHandler(HttpServerErrorException e) {
        Result<Void> result = Result.error(e);
        if (StrUtil.isNotBlank(apiProperties.getSource()))
            result.setSource(apiProperties.getSource());
        log.error(result.toString(), e);
        return result;
    }

    /**
     * 所有AbstractException异常及其子异常的处理
     *
     * @param e
     * @return
     */
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    @ExceptionHandler(AbstractException.class)
    public Result<Void> abstractExceptionHandler(AbstractException e) {
        Result<Void> result = Result.error(e);
        if (StrUtil.isNotBlank(apiProperties.getSource()))
            result.setSource(apiProperties.getSource());
        log.error(result.toString(), e);
        return result;
    }

    /**
     * 所有AbstractRuntimeException异常及其子异常的处理
     *
     * @param e
     * @return
     */
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    @ExceptionHandler(AbstractRuntimeException.class)
    public Result<Void> abstractRuntimeException(AbstractRuntimeException e) {
        Result<Void> result = Result.error(e);
        if (StrUtil.isNotBlank(apiProperties.getSource()))
            result.setSource(apiProperties.getSource());
        log.error(result.toString(), e);
        return result;
    }

    /**
     * 所有RuntimeException异常的处理
     * 8000+
     *
     * @param e
     * @return
     */
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    @ExceptionHandler(RuntimeException.class)
    public Result<Void> runtimeExceptionHandler(RuntimeException e) {
        Result<Void> result = Result.error(e);
        if (StrUtil.isNotBlank(apiProperties.getSource()))
            result.setSource(apiProperties.getSource());
        log.error(result.toString(), e);
        return result;
    }

    /**
     * 所有Exception异常的处理
     * 9000+
     *
     * @param e
     * @return
     */
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    @ExceptionHandler(Exception.class)
    public Result<Void> exceptionHandler(Exception e) {
        Result<Void> result = Result.error(e);
        if (StrUtil.isNotBlank(apiProperties.getSource()))
            result.setSource(apiProperties.getSource());
        log.error(result.toString(), e);
        return result;
    }


    //    /**
//     * 所有FailureException异常的处理
//     * 1000+
//     *
//     * @param e
//     * @return
//     */
//    @ExceptionHandler(FailureException.class)
//    public Result<String> failureExceptionHandler(FailureException e) {
//        Result<String> result = Result.error(e);
//        log.error(result.toString(), e);
//        return result;
//    }
//
//    /**
//     * 所有VerificationException异常的处理
//     * 2000+
//     *
//     * @param e
//     * @return
//     */
//    @ExceptionHandler(VerificationException.class)
//    public Result<String> verificationExceptionHandler(VerificationException e) {
//        Result<String> result = Result.error(e);
//        log.error(result.toString(), e);
//        return result;
//    }
//
//    /**
//     * 所有DataFailureException异常的处理
//     * 3000+
//     *
//     * @param e
//     * @return
//     */
//    @ExceptionHandler(DataFailureException.class)
//    public Result<String> dataFailureExceptionHandler(DataFailureException e) {
//        Result<String> result = Result.error(e);
//        log.error(result.toString(), e);
//        return result;
//    }
//
//
//    /**
//     * 所有DataException异常的处理
//     * 4000+
//     *
//     * @param e
//     * @return
//     */
//    @ExceptionHandler(DataException.class)
//    public Result<String> dataExceptionHandler(DataException e) {
//        Result<String> result = Result.error(e);
//        log.error(result.toString(), e);
//        return result;
//    }
//
//
//    /**
//     * 所有APIException异常的处理
//     * 5000+
//     *
//     * @param e
//     * @return
//     */
//    @ExceptionHandler(ApiException.class)
//    public Result<String> apiExceptionHandler(ApiException e) {
//        Result<String> result = Result.error(e);
//        log.error(result.toString(), e);
//        return result;
//    }
//
//    /**
//     * 所有DataBaseException异常的处理
//     * 6000+
//     *
//     * @param e
//     * @return
//     */
//    @ExceptionHandler(DataBaseException.class)
//    public Result<String> dataBaseExceptionHandler(DataBaseException e) {
//        Result<String> result = Result.error(e);
//        log.error(result.toString(), e);
//        return result;
//    }
//
//    /**
//     * 所有SysException异常的处理
//     * 7000+
//     *
//     * @param e
//     * @return
//     */
//    @ExceptionHandler(SysException.class)
//    public Result<String> sysExceptionHandler(SysException e) {
//        Result<String> result = Result.error(e);
//        log.error(result.toString(), e);
//        return result;
//    }
//

    //    @ResponseStatus(code = HttpStatus.INTERNAL_SERVER_ERROR)//支持@ResponseStatus注解和HttpServletRequest，HttpServletResponse入参
//    public Result<Void> abstractExceptionHandler(AbstractException e, HttpServletRequest request, HttpServletResponse response) {

}
