package tech.xmagic.api;


import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.feiniaojin.gracefulresponse.api.ExcludeFromGracefulResponse;
import io.swagger.v3.oas.annotations.media.Schema;
import lombok.Data;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.HttpServerErrorException;
import tech.xmagic.core.ResultCode;
import tech.xmagic.enums.RC;
import tech.xmagic.exception.AbstractException;
import tech.xmagic.exception.AbstractRuntimeException;

import java.io.Serializable;
import java.util.Optional;

/**
 * API统一返回格式
 * JSON数据格式封装
 * HTTP状态码统一在异常那里处理
 * 成功：成功返回数据必须是json对象数据，禁止使用字符串
 * {
 * “code”:0,
 * "message":"操作成功",
 * "error":null,
 * "success":true,
 * "bizCode":0,
 * "data":{}
 * }
 * 失败:
 * {
 * “code”:XXX,
 * "message":"失败说明",
 * "error":"错误描述",
 * "success":false,
 * "bizCode":0,
 * "data":null
 * }
 *
 * @author meng2c
 * @version 2021.7.1
 * @since 2021.7.13
 */
@ExcludeFromGracefulResponse
@Data
@Schema(title = "返回结果", description = "统一API返回结果")
public class Result<T> implements Serializable {
    /**
     * 状态码，
     * <ul>
     *     <li>0代表响应成功</li>
     *     <li>非0代表响应失败</li>
     * </ul>
     */
    @Schema(name = "code", title = "状态码", description = "状态码", example = "0", required = true)
    private int code = 9999;
    /**
     * 响应信息
     * <ul>
     *     <li>成功，表示响应消息</li>
     *     <li>失败，表示错误消息</li>
     * </ul>
     */
    @Schema(title = "响应信息", description = "响应信息", example = "操作成功", required = true)
    private String message;

    /**
     * 错误描述
     * <ul>
     *     <li>成功，为空</li>
     *     <li>失败，表示错误详情</li>
     * </ul>
     */
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(title = "错误描述", description = "错误描述")
    private String error;
    /**
     * 返回结果的目标来源
     */
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(title = "来源", description = "来源")
    private String source;

    /**
     * 业务码
     */
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(name = "tag", title = "标签", description = "标签", example = "0")
    private String tag;
    /**
     * 通过code来判断是否响应成功
     */
    @JsonIgnore
    @Schema(title = "是否响应成功", description = "是否响应成功", example = "true")
    private boolean success;


    /**
     * 响应的具体数据
     */
    @JsonInclude(JsonInclude.Include.NON_NULL)
    @Schema(title = "响应数据", description = "响应数据")
    private T data;

    /**
     * 无参构造
     */
    private Result() {
    }
    /********** 私有构造函数 **********/
    /**
     * 返回结果(全属性)
     *
     * @param code    状态码
     * @param message 消息
     * @param error   错误详情
     * @param source  来源
     * @param tag     标签
     * @param data    数据
     */
    private Result(int code, String message, String error, String source, String tag, T data) {
        this.code = code;
        this.message = message;
        this.error = error;
        this.source = source;
        this.tag = tag;
        this.data = data;
        this.success = RC.SUCCESS.getCode() == code;
    }

    /**
     * 返回结果(无标签)
     *
     * @param code    状态码
     * @param message 消息
     * @param error   错误详情
     * @param source  来源
     * @param data    数据
     */
    private Result(int code, String message, String error, String source, T data) {
        this(code, message, error, source, null, data);
    }

    /**
     * 返回结果(无来源,无标签)
     *
     * @param code    状态码
     * @param message 消息
     * @param error   错误详情
     * @param data    数据
     */
    private Result(int code, String message, String error, T data) {
        this(code, message, error, null, null, data);
    }

    /********** 公有构造函数,开发调用 **********/
    /**
     * 返回结果(全属性)
     *
     * @param resultCode 状态枚举
     * @param error      错误详情
     * @param source     来源
     * @param tag        标签
     * @param data       数据
     */
    public Result(ResultCode resultCode, String error, String source, String tag, T data) {
        this(resultCode.getCode(), resultCode.getMsg(), error, source, tag, data);
    }

    /**
     * 返回结果(无标签)
     *
     * @param resultCode 状态枚举
     * @param error      错误详情
     * @param source     来源
     * @param data       数据
     */
    public Result(ResultCode resultCode, String error, String source, T data) {
        this(resultCode.getCode(), resultCode.getMsg(), error, source, null, data);
    }

    /**
     * 返回结果(无来源,无标签)
     *
     * @param resultCode 状态枚举
     * @param error      错误详情
     * @param data       数据
     */
    public Result(ResultCode resultCode, String error, T data) {
        this(resultCode, error, null, null, data);
    }


    /**
     * 返回结果(无详细错误,无来源,无业务码,含数据)
     *
     * @param resultCode 状态
     * @param data       数据
     */
    public Result(ResultCode resultCode, T data) {
        this(resultCode, null, null, null, data);
    }

    /**
     * 返回结果(无详细错误,无来源,无业务码,无数据)
     *
     * @param resultCode
     */
    public Result(ResultCode resultCode) {
        this(resultCode, null, null, null, null);
    }


    /**
     * 判断是否成功
     *
     * @return 是否成功
     */
    public boolean isSuccess() {
        return RC.SUCCESS.getCode() == code;
    }

    /**
     * 判断结果是否失败
     *
     * @return 是否失败
     */
    @JsonIgnore
    public boolean isFailure() {
        return !isSuccess();
    }

    @Override
    public String toString() {
        return String.format("Result(success=%s,code=%d,msg=%s,err=%s,source=%s,tag=%s,data=%s)", this.isSuccess(), this.getCode(), this.getMessage(), this.getError(), this.getSource(), this.getTag(), this.getData());
    }

    /********** 公有静态方法 **********/
    /**
     * 判断结果是否成功
     *
     * @param result
     * @return
     */
    public static boolean isSuccess(Result<?> result) {
        return Optional.ofNullable(result).map((x) -> RC.SUCCESS.getCode() == x.code).orElse(Boolean.FALSE);
    }

    /**
     * 判断结果是否失败
     *
     * @param result
     * @return
     */
    public static boolean isFailure(Result<?> result) {
        return !isSuccess(result);
    }

    /***** 成功数据返回*****/
    /**
     * 完整属性
     *
     * @param source 来源
     * @param tag    标签
     * @param data   数据
     * @param <T>    数据
     * @return 返回结果对象
     */
    public static <T> Result<T> success(String source, String tag, T data) {
        return new Result<>(RC.SUCCESS, null, source, tag, data);
    }

    /**
     * 带数据的成功返回
     *
     * @param data 数据
     * @return 返回结果对象
     */
    public static <T> Result<T> success(String source, T data) {
        return success(source, null, data);
    }

    /**
     * 带数据的成功返回
     *
     * @param data 数据
     * @return 返回结果对象
     */
    public static <T> Result<T> success(T data) {
        return success(null, null, data);
    }
    //TODO 尝试用Void代替无数据返回的T泛型


    /**
     * 成功返回
     *
     * @return 返回结果对象
     */
    public static Result<Void> success() {
        return success(null);
    }


    /*****失败类结果*****/
    /**
     * 完整属性
     *
     * @param resultCode 返回状态枚举
     * @param error      错误描述
     * @return
     */
    public static Result<Void> failure(ResultCode resultCode, String error, String source, String tag) {
        return new Result<>(resultCode, error, source, tag, null);
    }

    public static Result<Void> failure(String error, String source, String tag) {
        return failure(RC.FAILURE, error, source, tag);
    }

    /**
     * 仅含错误消息
     *
     * @param resultCode
     * @param error
     * @return
     */
    public static Result<Void> failure(ResultCode resultCode, String error, String source) {
        return failure(resultCode, error, source, null);
    }

    public static Result<Void> failure(ResultCode resultCode, String error) {
        return failure(resultCode, error, null, null);
    }

    /**
     * 仅错误
     *
     * @param resultCode 返回状态枚举
     * @return
     */
    public static Result<Void> failure(ResultCode resultCode) {
        return failure(resultCode, null);
    }

    public static Result<Void> failure(String error) {
        return failure(RC.FAILURE, error);
    }


    public static Result<Void> failure() {
        return failure(RC.FAILURE, null);
    }

    /*****错误类结果*****/
    /**
     * 完整属性
     *
     * @param resultCode 返回状态枚举
     * @param error      错误描述
     * @return
     */
    public static Result<Void> error(ResultCode resultCode, String error, String source, String tag) {
        return new Result<>(resultCode, error, source, tag, null);
    }

    public static Result<Void> error(ResultCode resultCode, String error, String source) {
        return error(resultCode, error, source, null);
    }

    public static Result<Void> error(ResultCode resultCode, String error) {
        return error(resultCode, error, null);
    }

    public static Result<Void> error(ResultCode resultCode) {
        return error(resultCode, null);
    }

    public static Result<Void> error(String error, String source, String tag) {
        return new Result<>(RC.RUNTIME_EXCEPTION, error, source, tag, null);
    }

    public static Result<Void> error(String error, String source) {
        return error(error, source, null);
    }

    public static Result<Void> error(String error) {
        return error(error, null);
    }

    public static Result<Void> error() {
        return error("");
    }

    /**
     * Exception 错误
     *
     * @param e Exception类型的异常
     * @return
     */
    public static Result<Void> error(Exception e) {
        return new Result<>(RC.EXCEPTION, e.getMessage(), null);
    }

    /**
     * RuntimeException 错误
     *
     * @param e
     * @return
     */
    public static Result<Void> error(RuntimeException e) {
        return new Result<>(RC.RUNTIME_EXCEPTION, e.getMessage(), null);
    }

    /**
     * AbstractException 及其子类错误
     *
     * @param e
     * @return
     */
    public static Result<Void> error(AbstractException e) {
        return new Result<>(e.getCode(), e.getMsg(), e.getMessage(), null, e.getTag(), null);
    }

    /**
     * AbstractRuntimeException 及其子类错误
     *
     * @param e
     * @return
     */
    public static Result<Void> error(AbstractRuntimeException e) {
        return new Result<>(e.getCode(), e.getMsg(), e.getMessage(), null, e.getTag(), null);
    }

    /**
     * HttpClientErrorException 错误
     *
     * @param e
     * @return
     */
    public static Result<Void> error(HttpClientErrorException e) {
        //TODO 有待测试确定错误来源
        return new Result<>(RC.API_RPC_EXCEPTION, e.getMessage(), null);
    }

    /**
     * HttpServerErrorException 错误
     *
     * @param e
     * @return
     */
    public static Result<Void> error(HttpServerErrorException e) {
        //TODO 有待测试确定错误来源
        return new Result<>(RC.API_RPC_EXCEPTION, e.getMessage(), null);
    }


}
