package tech.xmagic.api;


import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.swagger.v3.oas.annotations.media.Schema;
import lombok.Data;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.HttpServerErrorException;
import tech.xmagic.core.ResultCode;
import tech.xmagic.enums.RC;
import tech.xmagic.exception.AbstractException;
import tech.xmagic.exception.AbstractRuntimeException;

import java.io.Serializable;
import java.util.Optional;

/**
 * API统一返回格式
 * JSON数据格式封装
 * HTTP状态码统一在异常那里处理
 * 成功：成功返回数据必须是json对象数据，禁止使用字符串
 * {
 * “code”:0,
 * "message":"操作成功",
 * "error":null,
 * "success":true,
 * "data":{}
 * }
 * 失败:
 * {
 * “code”:XXX,
 * "message":"失败说明",
 * "error":"错误描述",
 * "success":false,
 * "data":null
 * }
 *
 * @author meng2c
 * @version 2021.7.1
 * @since 2021.7.13
 */
@Data
//@JsonIgnoreProperties(value = {"httpStatus"})
@Schema(title = "返回结果", description = "统一API返回结果")
public class R<T>  implements Serializable{
    /**
     * 状态码，
     * <ul>
     *     <li>0代表响应成功</li>
     *     <li>非0代表响应失败</li>
     * </ul>
     */
    @Schema(name = "code", title = "状态码", description = "状态码", example = "0", required = true)
    private int code;
    /**
     * 响应信息
     * <ul>
     *     <li>成功，表示响应消息</li>
     *     <li>失败，表示错误消息</li>
     * </ul>
     */
    @Schema(title = "响应信息", description = "响应信息", example = "操作成功", required = true)
    private String message;

    /**
     * 错误描述
     * <ul>
     *     <li>成功，为空</li>
     *     <li>失败，表示错误详情</li>
     * </ul>
     */
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(title = "错误描述", description = "错误描述")
    private String error;
    /**
     * 返回结果的目标来源
     */
    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @Schema(title = "来源", description = "来源")
    private String source;
    /**
     * 通过code来判断是否响应成功
     */
    @JsonIgnore
    @Schema(title = "是否响应成功", description = "是否响应成功", example = "true", required = true)
    private boolean success;

    /**
     * 响应的具体数据
     */
    @JsonInclude(JsonInclude.Include.NON_NULL)
    @Schema(title = "响应数据", description = "响应数据")
    private T data;

    /********** 私有构造函数 **********/
    /**
     * 返回结果(全属性)
     *
     * @param code    状态码
     * @param message 消息
     * @param error   错误详情
     * @param source  来源
     * @param data    数据
     */
    private R(int code, String message, String error, String source, T data) {
        this.code = code;
        this.message = message;
        this.error = error;
        this.source = source;
        this.success = RC.SUCCESS.getCode() == code;
        this.data = data;
    }

    /**
     * 返回结果(无来源,HTTP状态码200)
     *
     * @param code    状态码
     * @param message 消息
     * @param error   错误详情
     * @param data    数据
     */
    private R(int code, String message, String error, T data) {
        this(code, message, error, null, data);
    }

    /********** 公有构造函数,开发调用 **********/
    /**
     * 返回结果(全属性)
     *
     * @param resultCode 状态枚举
     * @param error      错误详情
     * @param source     来源
     * @param data       数据
     */
    public R(ResultCode resultCode, String error, String source, T data) {
        this(resultCode.getCode(), resultCode.getMsg(), error, source, data);
    }

    /**
     * 返回结果(无来源)
     *
     * @param resultCode 状态枚举
     * @param error      错误详情
     * @param data       数据
     */
    public R(ResultCode resultCode, String error, T data) {
        this(resultCode.getCode(), resultCode.getMsg(), error, data);
    }


    /**
     * 返回结果(成功,无来源,含数据)
     *
     * @param resultCode 状态
     * @param data       数据
     */
    public R(ResultCode resultCode, T data) {
        this(resultCode, null, data);
    }

    /**
     * 返回结果(成功,无来源,无数据)
     *
     * @param resultCode
     */
    public R(ResultCode resultCode) {
        this(resultCode, null);
    }

    /**
     * 无参构造
     */
    private R() {
    }

    public boolean isSuccess() {
        return RC.SUCCESS.getCode() == code;
    }

    /**
     * 判断结果是否失败
     *
     * @return
     */
    @JsonIgnore
    public boolean isFailure() {
        return !isSuccess();
    }

    @Override
    public String toString() {
        return String.format("Result(success=%s,code=%d,msg=%s,err=%s,data=%s)", this.isSuccess(), this.getCode(), this.getMessage(), this.getError(), this.getData());
    }

    /********** 公有静态方法 **********/
    /**
     * 判断结果是否成功
     *
     * @param result
     * @return
     */
    public static boolean isSuccess(R<?> result) {
        return Optional.ofNullable(result).map((x) -> RC.SUCCESS.getCode() == x.code).orElse(Boolean.FALSE);
    }

    /**
     * 判断结果是否失败
     *
     * @param result
     * @return
     */
    public static boolean isFailure(R<?> result) {
        return !isSuccess(result);
    }

    /***** 成功数据返回*****/
    /**
     * 成功_x，含数据，无错误描述
     * <p>成功不允许自定义状态码，限定为0</p>
     *
     * @param resultCode 返回状态枚举
     * @param data       数据
     * @return
     */
    private static <T> R<T> success(ResultCode resultCode, T data) {
        return new R<>(resultCode, data);
    }

    /**
     * 成功_0，含数据，无错误描述
     *
     * @param data 数据
     * @return
     */
    public static <T> R<T> success(T data) {
        return success(RC.SUCCESS, data);
    }
    //TODO 尝试用Void代替无数据返回的T泛型

    /**
     * 成功_x，无数据，无错误描述
     * <p>成功不允许自定义状态码，限定为0</p>
     *
     * @param resultCode
     * @return
     */
    private static R<Void> success(ResultCode resultCode) {
        return new R<>(resultCode);
    }

    /**
     * 成功_0，无数据，无错误描述
     *
     * @return
     */
    public static R<Void> success() {
        return success(RC.SUCCESS);
    }


    /*****失败类结果*****/
    /**
     * 失败_x，无数据，含错误描述
     *
     * @param resultCode 返回状态枚举
     * @param error      错误描述
     * @return
     */
    public static R<Void> failure(ResultCode resultCode, String error) {
        return new R<>(resultCode, error, null);
    }

    /**
     * 失败_1000，无数据，含错误描述
     *
     * @param error 错误描述
     * @return
     */
    public static R<Void> failure(String error) {
        return failure(RC.FAILURE, error);
    }

    /**
     * 失败_x，无数据，无错误描述
     *
     * @param resultCode 返回状态枚举
     * @return
     */
    public static R<Void> failure(ResultCode resultCode) {
        return failure(resultCode, null);
    }

    /**
     * 失败_1000，无数据，无错误描述
     *
     * @return
     */
    public static R<Void> failure() {
        return failure(RC.FAILURE, null);
    }

    /*****错误类结果*****/
    /**
     * 错误_x，无数据，含错误描述
     *
     * @param resultCode 返回状态枚举
     * @param error      错误描述
     * @return
     */
    public static R<Void> error(ResultCode resultCode, String error) {
        return new R<>(resultCode, error, null);
    }
    /**
     * 错误_x，无数据，无错误描述
     *
     * @param resultCode 返回状态枚举
     * @return
     */
    public static R<Void> error(ResultCode resultCode) {
        return error(resultCode, null);
    }
    /**
     * Exception 错误
     *
     * @param e Exception类型的异常
     * @return
     */
    public static R<Void> error(Exception e) {
        return new R<>(RC.EXCEPTION, e.getMessage(), null);
    }

    /**
     * RuntimeException 错误
     *
     * @param e
     * @return
     */
    public static R<Void> error(RuntimeException e) {
        return new R<>(RC.RUNTIME_EXCEPTION, e.getMessage(), null);
    }

    /**
     * AbstractException 及其子类错误
     *
     * @param e
     * @return
     */
    public static R<Void> error(AbstractException e) {
        return new R<>(e.getCode(), e.getMsg(), e.getMessage(), null);
    }

    /**
     * AbstractRuntimeException 及其子类错误
     *
     * @param e
     * @return
     */
    public static R<Void> error(AbstractRuntimeException e) {
        return new R<>(e.getCode(), e.getMsg(), e.getMessage(), null);
    }

    /**
     * HttpClientErrorException 错误
     *
     * @param e
     * @return
     */
    public static R<Void> error(HttpClientErrorException e) {
        //TODO 有待测试确定错误来源
        return new R<>(RC.API_RPC_EXCEPTION, e.getMessage(), null);
    }

    /**
     * HttpServerErrorException 错误
     *
     * @param e
     * @return
     */
    public static R<Void> error(HttpServerErrorException e) {
        //TODO 有待测试确定错误来源
        return new R<>(RC.API_RPC_EXCEPTION, e.getMessage(), null);
    }


}
