/*
 *
 *  Copyright (c) 2020.
 *  This system ist developed by Jes Müller and Quirin Brändli!
 *  All rights reserved!
 *
 *  Please read the licence for more information.
 *
 */

package de.kelanisystem.kelanilogger.util;

import de.kelanisystem.kelanilogger.KelaniLogger;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;

public class FileUtil {
    private static final int MAX_NUMBER_OF_LOGS = 5;
    private final KelaniLogger kelaniLogger;

    public FileUtil(final KelaniLogger kelaniLogger) {
        this.kelaniLogger = kelaniLogger;
    }

    public void createFileWithFolder(final File logFile) throws IOException {
        if (!logFile.getParentFile().exists() && !logFile.getParentFile().mkdirs())
            throw new IOException("Couldn't create parent file: " + logFile.getCanonicalPath());
        if (!logFile.exists() && !logFile.createNewFile())
            throw new IOException("Couldn't create file: " + logFile.getCanonicalPath());
    }

    public void clearOldLogs(final File dir) throws IOException {
        final File directory = new File(dir, KelaniLogger.FILE_SEPARATOR);

        if (directory.exists()) {
            final List<File> files = new CopyOnWriteArrayList<>(Arrays.asList(Objects.requireNonNull(directory.listFiles())));

            //Sorts with modification, the oldest first
            files.sort(Comparator.comparingLong(File::lastModified));

            //Removes all files older than 7 days and everything with .lck as filetype
            removeOldFilesAndLocks(files);

            //Deletes all remaining files, if more than 5 present. Oldest first.
            deleteAllRemainingFiles(files);
        }
    }

    private void removeOldFilesAndLocks(final List<File> files) throws IOException {
        for (final File file : files) {
            if (file.lastModified() <= (System.currentTimeMillis() - TimeUnit.DAYS.toMillis(7))) {
                if (!file.delete() && files.remove(file))
                    throw new IOException("Couldn't delete directory: " + file.getCanonicalPath());
            } else if (file.getName().contains(".lck")) {
                files.remove(file);
            }
        }
    }

    private void deleteAllRemainingFiles(final List<File> files) throws IOException {
        if (files.size() >= MAX_NUMBER_OF_LOGS) {
            for (int i = 0; i < files.size(); i++) {
                if (files.size() > MAX_NUMBER_OF_LOGS) {
                    if (!files.get(i).delete())
                        throw new IOException("Couldn't delete directory: " + files.get(i).getCanonicalPath());
                    files.remove(i);
                }
            }
        }
    }
}
