/*
 *
 *  Copyright (c) 2020.
 *  This system ist developed by Jes Müller and Quirin Brändli!
 *  All rights reserved!
 *
 *  Please read the licence for more information.
 *
 */

package de.kelanisystem.kelanilogger.formats;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.logging.Formatter;
import java.util.logging.Level;
import java.util.logging.LogRecord;

import static java.util.logging.Level.*;

public class DefaultFormat extends Formatter {
    private static final boolean IS_WINDOWS = System.getProperty("os.name").toUpperCase(Locale.US).contains("WINDOWS");

    private static final String ANSI_RESET = "\u001B[0m";
    private static final String ANSI_RED = "\u001B[31m";
    private static final String ANSI_GREEN = "\u001B[32m";
    private static final String ANSI_YELLOW = "\u001B[33m";
    private static final String ANSI_WHITE = "\u001B[37m";
    private static final String ANSI_CYAN = "\u001B[36m";

    @Override
    public String format(final LogRecord logRecord) {
        final StringBuilder builder = new StringBuilder();

        final Level level = logRecord.getLevel();

        if (!IS_WINDOWS)
            builder.append(getLevelColor(logRecord.getLevel()));


        if (level != OFF) {
            builder.append('[')
                    .append(calcDate(logRecord.getMillis()))
                    .append(']')

                    .append(" [")
                    .append(logRecord.getLevel().getName())
                    .append(']');

            if (!IS_WINDOWS)
                builder.append(ANSI_WHITE);
            builder.append(" - ");
        }

        builder.append(logRecord.getMessage());

        final Object[] params = logRecord.getParameters();

        if (params != null) {
            builder.append('\t');
            for (int i = 0; i < params.length; i++) {
                builder.append(params[i]);
                if (i < params.length - 1)
                    builder.append(", ");
            }
        }

        builder.append(getAnsiReset())
                .append('\n');

        return builder.toString();
    }

    private String getAnsiReset() {
        return IS_WINDOWS ? "" : ANSI_RESET;
    }

    private String getLevelColor(final Level level) {
        String returnStatement = ANSI_WHITE;

        if (INFO.equals(level)) {
            returnStatement = ANSI_CYAN;
        } else if (SEVERE.equals(level)) {
            returnStatement = ANSI_RED;
        } else if (WARNING.equals(level)) {
            returnStatement = ANSI_YELLOW;
        } else if (FINE.equals(level)) {
            returnStatement = ANSI_GREEN;
        }
        return returnStatement;
    }

    private String calcDate(final long milliseconds) {
        // de_DE
        final SimpleDateFormat dateFormat = new SimpleDateFormat("dd-MM-yyyy HH:mm:ss");
        final Date date = new Date(milliseconds);
        return dateFormat.format(date);
    }
}
