/*
 *
 *  Copyright (c) 2020.
 *  This system ist developed by Jes Müller and Quirin Brändli!
 *  All rights reserved!
 *
 *  Please read the licence for more information.
 *
 */

package de.kelanisystem.kelanilogger;

import de.kelanisystem.kelanilogger.config.KelaniLoggerConfig;
import de.kelanisystem.kelanilogger.exceptions.NoFileSpecifiedException;
import de.kelanisystem.kelanilogger.formats.DefaultFormat;
import de.kelanisystem.kelanilogger.handler.KelaniConsoleHandler;
import de.kelanisystem.kelanilogger.util.FileUtil;

import java.io.File;
import java.io.IOException;
import java.util.Date;
import java.util.List;
import java.util.logging.ConsoleHandler;
import java.util.logging.FileHandler;
import java.util.logging.Level;
import java.util.logging.Logger;


public class KelaniLogger {
    public static final String FILE_SEPARATOR = System.getProperty("file.separator");
    private Logger logger;
    private final FileUtil fileUtil;

    public KelaniLogger(final KelaniLoggerConfig kelaniLoggerConfig) {
        fileUtil = new FileUtil(this);
        createLogger(kelaniLoggerConfig);
        createFileLogger(kelaniLoggerConfig);
        finish(kelaniLoggerConfig);
    }

    private void createLogger(final KelaniLoggerConfig kelaniLoggerConfig) {
        if(kelaniLoggerConfig.getLogger() != null) {
            logger = kelaniLoggerConfig.getLogger();
        } else {
            logger.setUseParentHandlers(false);
        }

        if (!kelaniLoggerConfig.getKelaniConsoleHandlers().isEmpty()) {
            for (final KelaniConsoleHandler kelaniConsoleHandler : kelaniLoggerConfig.getKelaniConsoleHandlers()) {
                if (!kelaniConsoleHandler.isFormatSet()) kelaniConsoleHandler.setFormatter(new DefaultFormat());
                if (!kelaniConsoleHandler.isLevelSet())
                    kelaniConsoleHandler.setLevel(kelaniLoggerConfig.getLogLevelConsole());
                logger.addHandler(kelaniConsoleHandler);
            }
        }
    }

    /* Old Method, may be used again
    private void createConsoleHandler(final KelaniLoggerConfig kelaniLoggerConfig) {
        if (!kelaniLoggerConfig.isDefaultConsoleHandlerDeactivated()) {
            consoleHandler = new ConsoleHandler();
            consoleHandler.setFormatter((kelaniLoggerConfig.getFormatter() == null) ? new DefaultFormat() : kelaniLoggerConfig.getFormatter());
            consoleHandler.setLevel(kelaniLoggerConfig.getLogLevelConsole());
            logger.addHandler(consoleHandler);
        }
    }
     */

    private void createFileLogger(final KelaniLoggerConfig kelaniLoggerConfig) {
        try {

            if (kelaniLoggerConfig.getDirectory() == null || kelaniLoggerConfig.getLogFileName() == null)
                throw new NoFileSpecifiedException();

            //Date for log file
            final Date date = new Date();

            final String logFileName = parseLogName(date, kelaniLoggerConfig);

            final File logFile = new File(logFileName);
            fileUtil.createFileWithFolder(logFile);

            setupFileHandler(kelaniLoggerConfig, logFileName);
            fileUtil.clearOldLogs(logFile.getParentFile());
        } catch (IOException | NoFileSpecifiedException e) {
            error(e.getMessage());
        }
    }

    private void setupFileHandler(final KelaniLoggerConfig kelaniLoggerConfig, final String logFileName) throws IOException {
        final FileHandler fileHandler = new FileHandler(logFileName, true);
        fileHandler.setFormatter(kelaniLoggerConfig.getFormatter());
        fileHandler.setLevel(kelaniLoggerConfig.getLogLevelFile());
        logger.addHandler(fileHandler);
    }

    private void finish(final KelaniLoggerConfig kelaniLoggerConfig) {
        if (kelaniLoggerConfig.getLogo() != null)
            off(kelaniLoggerConfig.getLogo());
    }

    private String parseLogName(final Date date, final KelaniLoggerConfig config) throws IOException {
        String filename = config.getLogFileName();
        filename = filename.replace("%LOGGER_NAME%", config.getLogFileName()).replace("%LOG_CREATE_TIME", date.toString());

        String name;
        if (config.getDirectory().getCanonicalPath().endsWith("log") || config.getDirectory().getCanonicalPath().endsWith("logs"))
            name = config.getDirectory().getCanonicalFile() + FILE_SEPARATOR + filename;
        else
            name = config.getDirectory().getCanonicalFile() + FILE_SEPARATOR + "logs" + FILE_SEPARATOR + filename;

        return name;
    }

    /**
     * Prints info into the logger.
     *
     * @param object The message to be send.
     */
    public void info(final Object object) {
        log(Level.INFO, object);
    }


    /**
     * Prints warning into the logger.
     *
     * @param object The message to be send.
     */
    public void warning(final Object object) {
        log(Level.WARNING, object);
    }

    /**
     * Prints error into the logger.
     *
     * @param object The message to be send.
     */
    public void error(final Object object) {
        log(Level.SEVERE, object);
    }

    /**
     * Prints message only into file, not console.
     *
     * @param object The message to be send.
     */
    public void config(final Object object) {
        log(Level.INFO, object);
    }


    /**
     * Prints message into log, with given level.
     *
     * @param level  Level of the message.
     * @param object The message to be send.
     */
    public void log(final Level level, final Object object) {
        if (logger.isLoggable(level))
            logger.log(Level.INFO, object::toString);
    }

    /**
     * Prints message as fine into the log.
     *
     * @param object The message to be send.
     */
    public void fine(final Object object) {
        log(Level.FINE, object);
    }

    /**
     * Prints message without any prefix and timestamp.
     *
     * @param object The message to be send.
     */
    public void off(final Object object) {
        log(Level.OFF, object);
    }

    /**
     * Print multiple messages as one level into the logger and file.
     *
     * @param level   The level, that should be printed.
     * @param objects The message to be send.
     */
    public void logMany(final Level level, final List<Object> objects) {
        final StringBuilder stringBuilder = new StringBuilder();

        for (int i = 0; i < objects.size(); i++) {
            stringBuilder.append(objects.get(i));
            if (i != (objects.size() - 1)) stringBuilder.append('\n');
        }

        log(level, stringBuilder.toString());
    }

    /**
     * Add a console handler to the logger.
     *
     * @param handlers handlers that should be added.
     */
    public void addHandlers(final ConsoleHandler... handlers) {
        for (final ConsoleHandler handler : handlers)
            logger.addHandler(handler);
    }


    /**
     * Removes a console handler to the logger.
     *
     * @param handlers handlers that should be removed.
     */
    public void removeHandlers(final ConsoleHandler... handlers) {
        for (final ConsoleHandler handler : handlers)
            logger.addHandler(handler);
    }
}
