package tech.poool.engage.core

import java.util.Calendar
import android.annotation.SuppressLint
import android.app.Activity
import android.content.Context
import android.view.ViewGroup
import android.view.ViewTreeObserver.OnScrollChangedListener
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.ui.platform.ComposeView
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.channels.BufferOverflow
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.launch
import org.jetbrains.annotations.TestOnly
import tech.poool.commons.helpers.pxToDp
import tech.poool.engage.BuildConfig
import tech.poool.engage.EngageConfig
import tech.poool.engage.EngageLogger
import tech.poool.engage.EngageTexts
import tech.poool.engage.EngageVariables
import tech.poool.engage.Engage
import tech.poool.engage.EngageEvents
import tech.poool.engage.compose.Element
import tech.poool.engage.EngageDb
import tech.poool.engage.EngageEventsManager
import tech.poool.engage.core.shared.LocalViewModel
import tech.poool.engage.network.EngageNetwork
import tech.poool.engage.network.responses.EngageElement
import tech.poool.engage.network.responses.EngageElementDelayCondition
import tech.poool.engage.network.responses.EngageElementPriorityCondition
import tech.poool.engage.network.responses.EngageElementScrollCondition
import tech.poool.engage.network.responses.EngageElementTargetCondition
import tech.poool.engage.network.responses.EngageElementViewsCondition
import tech.poool.engage.network.responses.EngageElementViewsPerDayCondition
import tech.poool.engage.network.responses.EngageElementVisitsCondition
import tech.poool.engage.network.responses.EngageElementVisitsPerDayCondition

internal class EngageManager(
    private val appId: String,
    private val mode: String? = Engage.MODE_VIEWS,
    private val context: Context? = null
) {
    private val config: EngageConfig by lazy { EngageConfig() }
    private val variables: EngageVariables by lazy { EngageVariables() }
    private val events: EngageEventsManager
    private val logger: EngageLogger
    private val api: EngageNetwork
    internal val texts: EngageTexts
    private val db: EngageDb? = context?.let { EngageDb(it) }

    internal val engageViewModel: EngageViewModel

    init {
        config.setVal("appId", appId, true)
        logger = EngageLogger(config = config)
        api = EngageNetwork(config = config, logger = logger, baseUrl = BuildConfig.API_URL)
        events = EngageEventsManager(logger = logger)
        texts = EngageTexts(config = config, variables = variables)

        engageViewModel = EngageViewModel(
            EngageDataConfig(
                config = config,
                variables = variables,
                texts = texts,
                events = events,
                logger = logger,
                api = api,
                db = db,
            )
        )
    }

    fun createElement(
        slug: String,
        target: ViewGroup? = null,
        onDisplay: ((element: EngageElement) -> Unit)? = null,
    ) {
        engageViewModel.viewModelScope.launch {
            try {
                val result = engageViewModel.getElement(slug).await()

                when (mode) {
                    Engage.MODE_VIEWS -> result?.element?.let { element ->
                        target?.let {
                            val composeView = ComposeView(target.context).apply {
                                setContent {
                                    CompositionLocalProvider(
                                        LocalViewModel provides engageViewModel,
                                    ) {
                                        Element(element = element)
                                    }
                                }
                                layoutParams = ViewGroup.LayoutParams(
                                    ViewGroup.LayoutParams.MATCH_PARENT,
                                    ViewGroup.LayoutParams.WRAP_CONTENT
                                )
                            }

                            logger.d("Creating element <${element.slug}>")

                            target.addView(composeView)
                        }
                    }

                    Engage.MODE_COMPOSE -> result?.element?.let { element ->
                        onDisplay?.invoke(element)
                    }
                }
            } catch (e: Exception) {
                logger.e("Error creating element")
                e.printStackTrace()
                events.emitSync(EngageEvents.ERROR, mapOf(
                    "error" to e.message,
                    "element" to mapOf(
                        "slug" to slug
                    )
                ))
            }
        }
    }

    @SuppressLint("DiscouragedApi")
    fun autoCreate(
        filters: List<String>? = null,
        scrollTarget: ViewGroup? = null,
        scrollFlow: MutableSharedFlow<Int>? = null,
        scrollHeight: Float? = null,
        onDisplay: ((targetId: String, element: EngageElement) -> Unit)? = null,
        onHide: ((targetId: String) -> Unit)? = null,
    ) {
        engageViewModel.viewModelScope.launch {
            try {
                val result = engageViewModel.autoCreate(filters).await()
                engageViewModel.updateFilters(filters)

                logger.d("Found elements: ${result?.elements}")

                val elements = (result?.elements ?: listOf()).sortedBy { getPriority(it) }

                elements.forEach { rawElement ->
                    val conditions = rawElement.conditions?.let { conditions ->
                        conditions.groupBy { c -> c.type }
                    } ?: mapOf()

                    val targetId =
                        conditions["target"]?.filterIsInstance<EngageElementTargetCondition>()
                            ?.get(0)?.value ?: (
                                // If no target is found but element has a bottom-sheet or modal condition,
                                // we use the "global container" -> a special id that will be detected by
                                // EngageElements() and displayed directly on compose, and displayed inside
                                // the root view in view-based apps
                                conditions["bottom-sheet"] ?: conditions["modal"]
                                )?.let { GLOBAL_CONTAINER_ID }

                    if (targetId.isNullOrEmpty()) {
                        logger.w("No target found, skipping element <${rawElement.slug}>")
                        return@forEach
                    }

                    // Visits
                    val visitsCondition = conditions["visits"]
                        ?.filterIsInstance<EngageElementVisitsCondition>()
                        ?.get(0)
                    val totalVisits = db?.getInt("pageViews.total") ?: 0

                    if ( // display
                        visitsCondition != null &&
                        visitsCondition.hide == null &&
                        totalVisits <= visitsCondition.value
                    ) {
                        logger.d("Minimum visits count not reached, skipping element <${rawElement.slug}>")
                        return@forEach
                    }

                    if ( // hide
                        visitsCondition != null &&
                        visitsCondition.hide == true &&
                        totalVisits > visitsCondition.value
                    ) {
                        logger.d("Maximum visits count reached, skipping element <${rawElement.slug}>")
                        return@forEach
                    }

                    // Visits per day
                    val visitsPerDayCondition = conditions["visitsPerDay"]
                        ?.filterIsInstance<EngageElementVisitsPerDayCondition>()
                        ?.get(0)
                    val totalVisitsPerDays = db
                        ?.getPerDay("pageViews", visitsPerDayCondition?.value?.days ?: 0) ?: 0

                    if ( // display
                        visitsPerDayCondition != null &&
                        visitsPerDayCondition.hide == null &&
                        totalVisitsPerDays <= visitsPerDayCondition.value.times
                    ) {
                        logger.d("Minimum visits per days not reached, skipping element <${rawElement.slug}>")
                        return@forEach
                    }

                    if ( // hide
                        visitsPerDayCondition != null &&
                        visitsPerDayCondition.hide == true &&
                        totalVisitsPerDays > visitsPerDayCondition.value.times
                    ) {
                        logger.d("Maximum visits per days reached, skipping element <${rawElement.slug}>")
                        return@forEach
                    }

                    // Views
                    val viewsCondition = conditions["views"]
                        ?.filterIsInstance<EngageElementViewsCondition>()
                        ?.get(0)
                    val totalViews = db?.getInt("views.${rawElement.id}.total") ?: 0

                    if (
                        viewsCondition != null &&
                        viewsCondition.hide == null &&
                        totalViews >= viewsCondition.value
                    ) {
                        logger.d("Views count limit reached, skipping element <${rawElement.slug}>")
                        return@forEach
                    }

                    // Views per day
                    val viewsPerDayCondition = conditions["viewsPerDay"]
                        ?.filterIsInstance<EngageElementViewsPerDayCondition>()
                        ?.get(0)
                    val totalViewsPerDays = db
                        ?.getPerDay(
                            "views.${rawElement.id}",
                            viewsPerDayCondition?.value?.days ?: 0
                        ) ?: 0

                    if (
                        viewsPerDayCondition != null &&
                        viewsPerDayCondition.hide == null &&
                        totalViewsPerDays >= viewsPerDayCondition.value.times
                    ) {
                        logger.d("Views per days limit reached, skipping element <${rawElement.slug}>")
                        return@forEach
                    }

                    // .autoCreate() only returns a list of active elements without content to save up
                    // bandwidth for elements that shouldn't be displayed right away, so we need to
                    // retrieve every element manually
                    val element = engageViewModel.getElement(rawElement.slug).await()?.element

                    element?.let {
                        if (conditions["delay"] != null || conditions["scroll"] != null) {
                            val flags = mutableMapOf(
                                "delay" to false,
                                "scroll" to false,
                                "destroyed" to false
                            )

                            val delayCondition = conditions["delay"]
                                ?.filterIsInstance<EngageElementDelayCondition>()
                                ?.getOrNull(0)

                            val scrollCondition = conditions["scroll"]
                                ?.filterIsInstance<EngageElementScrollCondition>()
                                ?.getOrNull(0)

                            if (delayCondition != null) {
                                logger.d(
                                    "Delay condition found, waiting for ${delayCondition.value}" +
                                            "before ${if (delayCondition.hide) "hiding" else "displaying"} " +
                                            "element <${rawElement.slug}}"
                                )

                                // Start a new coroutine so we don't block the upper level coroutine
                                // displaying other elements while we wait for this one
                                engageViewModel.viewModelScope.launch {
                                    delay(delayCondition.value.toLong() * 1000)

                                    logger.d(
                                        "Delay condition reached, " +
                                                "${if (delayCondition.hide) "hiding" else "displaying"} " +
                                                "element <${rawElement.slug}>"
                                    )

                                    if (
                                        scrollCondition == null ||
                                        flags["scroll"] == true ||
                                        scrollCondition.hide
                                    ) {
                                        if (flags["destroyed"] != true) {
                                            commitElementView(element)
                                        }

                                        if (delayCondition.hide) {
                                            if (
                                                flags["destroyed"] != true &&
                                                (scrollCondition?.hide != true || flags["scroll"] == true)
                                            ) {
                                                flags["destroyed"] = true
                                                destroyElement(targetId, element, onHide)
                                            }
                                        } else {
                                            renderElement(targetId, element, onDisplay)
                                        }
                                    }
                                }
                            }

                            if (scrollCondition != null) {
                                if (mode == Engage.MODE_VIEWS && scrollTarget == null) {
                                    logger.e(
                                        message = "A scrollTarget parameter is required on " +
                                                "the .autoCreate() method when using scroll conditions, " +
                                                "skipping element <${rawElement.slug}>"
                                    )

                                    return@forEach
                                } else if (mode == Engage.MODE_COMPOSE && scrollFlow == null) {
                                    logger.e(
                                        message = "A scrollState parameter is required on the " +
                                                "EngageElements() composable when using scroll " +
                                                "conditions, skipping element <${rawElement.slug}>"
                                    )

                                    return@forEach
                                } else {
                                    logger.d(
                                        message = "Scroll condition found, waiting for " +
                                                "${scrollCondition.value}% to be scrolled " +
                                                "before ${if (scrollCondition.hide) "hiding" else "displaying"} " +
                                                "element <${rawElement.slug}>"
                                    )
                                }

                                context?.let {
                                    val viewportHeight = pxToDp(
                                        context,
                                        context.resources.displayMetrics.heightPixels.toFloat()
                                    )
                                    val maxScroll = when (mode) {
                                        Engage.MODE_VIEWS -> pxToDp(
                                            context,
                                            scrollTarget?.getChildAt(0)?.height?.toFloat() ?: 0f
                                        )

                                        Engage.MODE_COMPOSE -> scrollHeight ?: 0f
                                        else -> 0f
                                    }
                                    val minScroll =
                                        maxScroll * (scrollCondition.value.toFloat() / 100)

                                    // We check if the viewport height is already greater than the scroll target
                                    // content height (scroll views always have a container as subview)
                                    if (viewportHeight >= maxScroll) {
                                        flags["scroll"] = true
                                        logger.d(
                                            "Scroll condition reached (viewport height <$viewportHeight> is " +
                                                    "greater than scroll target content height <$maxScroll>), " +
                                                    "${if (scrollCondition.hide) "hiding" else "displaying"} element <${rawElement.slug}>"
                                        )

                                        if (delayCondition == null || flags["delay"] == true || delayCondition.hide) {
                                            if (flags["destroyed"] != true) {
                                                commitElementView(element)
                                            }

                                            if (scrollCondition.hide) {
                                                if (
                                                    flags["destroyed"] != true &&
                                                    (delayCondition?.hide != true || flags["delay"] == true)
                                                ) {
                                                    flags["destroyed"] = true
                                                    destroyElement(targetId, element, onHide)
                                                } else {
                                                }
                                            } else {
                                                renderElement(targetId, element, onDisplay)
                                            }
                                        } else {
                                        }
                                    } else {
                                        val scrollFlowRef = scrollFlow ?: MutableSharedFlow(
                                            1,
                                            onBufferOverflow = BufferOverflow.DROP_OLDEST
                                        )

                                        var scrollListener: OnScrollChangedListener? = null

                                        engageViewModel.viewModelScope.launch {
                                            scrollFlowRef.collectLatest {
                                                if (flags["scroll"] == true) {
                                                    return@collectLatest
                                                }

                                                if (it >= minScroll) {
                                                    flags["scroll"] = true

                                                    if (delayCondition == null || flags["delay"] == true || delayCondition.hide) {
                                                        if (flags["destroyed"] != true) {
                                                            commitElementView(element)
                                                        }

                                                        if (scrollCondition.hide) {
                                                            if (
                                                                flags["destroyed"] != true &&
                                                                (delayCondition?.hide != true || flags["delay"] == true)
                                                            ) {
                                                                flags["destroyed"] = true
                                                                destroyElement(
                                                                    targetId,
                                                                    element,
                                                                    onHide
                                                                )
                                                            }
                                                        } else {
                                                            renderElement(
                                                                targetId,
                                                                element,
                                                                onDisplay
                                                            )
                                                            scrollTarget?.viewTreeObserver
                                                                ?.removeOnScrollChangedListener(
                                                                    scrollListener
                                                                )
                                                        }
                                                    }
                                                }
                                            }
                                        }

                                        scrollListener = OnScrollChangedListener {
                                            scrollFlowRef.tryEmit(scrollTarget?.scrollY ?: 0)
                                        }

                                        scrollTarget
                                            ?.viewTreeObserver
                                            ?.addOnScrollChangedListener(scrollListener)
                                    }
                                }
                            }

                            /*
                          If the conditons are just a combination of a delay and a scroll hide
                          we need to render the element that will be later destroyed
                        */
                            if (delayCondition?.hide == true && scrollCondition?.hide == true) {
                                renderElement(targetId, element, onDisplay)
                            }

                            /*
                          If the element has any of the following conditions, we need to
                          render it immediately if there is only one scroll or delay hide
                          condition
                        */
                            if (
                                (delayCondition?.hide == true && scrollCondition == null) ||
                                (scrollCondition?.hide == true && delayCondition == null)
                            ) {
                                renderElement(targetId, element, onDisplay)
                            }

                            return@forEach
                        }

                        // No scroll or delay conditions
                        commitElementView(element)
                        renderElement(targetId, element, onDisplay)
                    }
                }
            } catch (e: Exception) {
                logger.e("Error auto creating elements")
                e.printStackTrace()
                events.emitSync(EngageEvents.ERROR, mapOf("error" to e.message))
            }
        }
    }

    private fun renderElement (
        targetId: String,
        element: EngageElement?,
        onDisplay: ((targetId: String, element: EngageElement) -> Unit)?
    ) {
        when (mode) {
            Engage.MODE_VIEWS -> element?.let {
                try {
                    val resources = context?.resources
                    val view = (context as Activity).findViewById<ViewGroup>(
                        if (targetId == GLOBAL_CONTAINER_ID)
                            android.R.id.content
                        else
                            resources?.getIdentifier(
                                targetId,
                                "id",
                                context.packageName
                            ) ?: -1
                    )

                    if (view == null) {
                        logger.w("Target view $targetId not found, skipping element <${element.slug}>")
                        return@let
                    }

                    val composeView = ComposeView(context).apply {
                        setContent {
                            CompositionLocalProvider(
                                LocalViewModel provides engageViewModel,
                            ) {
                                Element(element = element)
                            }
                        }
                        layoutParams = ViewGroup.LayoutParams(
                            ViewGroup.LayoutParams.MATCH_PARENT,
                            ViewGroup.LayoutParams.WRAP_CONTENT
                        )
                        tag = element.id
                    }

                    logger.d("Creating element <${element.slug}>")
                    view.addView(composeView)
                } catch (e: Exception) {
                    logger.e("Error creating element <${element.slug}>: ${e.printStackTrace()}")
                }
            }
            Engage.MODE_COMPOSE -> element?.let {
                logger.d("Creating element <${element.slug}>")
                onDisplay?.invoke(targetId, element)
            }
        }
    }

    private fun destroyElement(
        targetId: String,
        element: EngageElement,
        onHide: ((targetId: String) -> Unit)? = null
    ) {
        when (mode) {
            Engage.MODE_VIEWS -> context?.let {
                val composeView = (context as Activity)
                    .findViewById<ViewGroup>(android.R.id.content)
                    .findViewWithTag<ComposeView>(element.id)

                (composeView?.parent as ViewGroup?)?.removeView(composeView)
            }
            Engage.MODE_COMPOSE -> {
                onHide?.invoke(targetId)
            }
        }
    }

    // Config
    fun config(key: String, value: Any?, readOnly: Boolean) =
        config.setVal(key, value, readOnly)

    fun config(mapConfig: Map<String, Any?>, readOnly: Boolean) {
        mapConfig.forEach { (key, value) ->
            config.setVal(key, value, readOnly)
        }
    }

    @TestOnly
    fun getConfig(): Map<String, Any?> = config.getAll()

    // Events
    fun on(
        event: EngageEvents,
        once: Boolean,
        callbackSync: ((Map<String, Any?>) -> Any?)? = null,
        callback: (suspend (Map<String, Any?>) -> Any?)? = null,
    ) {
        events.listen(event, once, callbackSync, callback)
    }

    fun off(event: EngageEvents) {
        events.mute(event)
    }

    suspend fun emit(event: EngageEvents, data: Map<String, Any?>): List<Any?> {
        return events.emit(event, data)
    }

    fun emitSync(event: EngageEvents, data: Map<String, Any?>): List<Any?> {
        return events.emitSync(event, data)
    }

    @TestOnly
    fun getEvents(): Map<EngageEvents, MutableList<Pair<Boolean, suspend (Map<String, Any?>) -> Any?>>> {
        return events.getAll()
    }

    // Variables
    fun variables(key: String, value: Any?) {
        variables.setVal(key, value)
    }

    fun variables(mapVariables: Map<String, Any?>) {
        mapVariables.forEach { (key, value) ->
            variables.setVal(key, value)
        }
    }

    @TestOnly
    fun getVariables(): Map<String, Any?> = config.getAll()

    // Texts
    fun texts(key: String, value: String?, readOnly: Boolean?, locale: String?) {
        texts.setVal(key, value, readOnly, locale)
    }

    fun texts(mapTexts: Map<String, String?>, readOnly: Boolean?, locale: String?) {
        mapTexts.forEach { (key, value) ->
            texts.setVal(key, value, readOnly, locale)
        }
    }

    fun commitPageView () {
        if (db == null) {
            logger.w("Cannot commit pageview, no context has been provided to Engage")
            return
        }

        engageViewModel.viewModelScope.launch {
            db.inc("pageViews.total")
            db.inc("pageViews.${db.formatDate(Calendar.getInstance())}")
        }
    }

    private fun commitElementView (element: EngageElement) {
        if (db == null) {
            logger.w("Cannot commit element view, no context has been provided to Engage")
            return
        }

        engageViewModel.viewModelScope.launch {
            db.inc("views.${element.id}.total")
            db.inc("views.${element.id}.${db.formatDate(Calendar.getInstance())}")
        }
    }

    @TestOnly
    fun getTexts(): Map<String, String?> = texts.getAll()

    private fun getPriority(element: EngageElement): Int {
        return element.conditions
            ?.filterIsInstance<EngageElementPriorityCondition>()
            ?.find { it.type == "priority" }
            ?.value ?: 0
    }

    companion object {
        internal const val GLOBAL_CONTAINER_ID = "poool-engage-global-container"
    }
}
