package tech.poool.engage.compose

import androidx.compose.foundation.ScrollState
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import kotlinx.coroutines.channels.BufferOverflow
import kotlinx.coroutines.flow.MutableSharedFlow
import tech.poool.engage.ClickEvent
import tech.poool.engage.DestroyEvent
import tech.poool.engage.ErrorEvent
import tech.poool.engage.FieldError
import tech.poool.engage.FormEvent
import tech.poool.engage.ReadyEvent
import tech.poool.engage.SeenEvent
import tech.poool.engage.core.EngageManager
import tech.poool.engage.core.shared.LocalEngage
import tech.poool.engage.network.responses.EngageElement
import tech.poool.engage.onClick
import tech.poool.engage.onDestroy
import tech.poool.engage.onError
import tech.poool.engage.onFormSubmit
import tech.poool.engage.onReady
import tech.poool.engage.onSeen

@Composable
fun EngageElements (
    filters: List<String>? = null,
    autoCommitPageView: Boolean? = false,
    scrollState: ScrollState? = null,
    onReady: ((ReadyEvent) -> Unit)? = null,
    onSeen: ((SeenEvent) -> Unit)? = null,
    onClick: ((ClickEvent) -> Unit)? = null,
    onFormSubmit: (suspend (FormEvent) -> List<FieldError>?)? = null,
    onDestroy: ((DestroyEvent) -> Unit)? = null,
    onError: ((ErrorEvent) -> Unit)? = null,
) {
    val (engage, targets) = LocalEngage.current
    var commitedPageview by remember { mutableStateOf(false) }
    val globalElements = remember { mutableStateListOf<EngageElement>() }
    val scrollFlow = remember { mutableStateOf<MutableSharedFlow<Int>>(
        MutableSharedFlow(1, onBufferOverflow = BufferOverflow.DROP_OLDEST)
    ) }

    LaunchedEffect(scrollState?.value) {
        scrollFlow.value.emit(scrollState?.value ?: 0)
    }

    LaunchedEffect(Unit) {
        engage
            ?.onReady { onReady?.invoke(it) }
            ?.onSeen { onSeen?.invoke(it) }
            ?.onClick { onClick?.invoke(it) }
            ?.onFormSubmit { return@onFormSubmit onFormSubmit?.invoke(it) }
            ?.onDestroy { onDestroy?.invoke(it) }
            ?.onError { onError?.invoke(it) }

        engage?.autoCreate(
            filters = filters,
            scrollFlow = scrollState?.let { scrollFlow.value },
            scrollHeight = scrollState?.maxValue?.toFloat() ?: 0f,
            onDisplay = { targetId, element ->
                if (targetId == EngageManager.GLOBAL_CONTAINER_ID) {
                    globalElements.add(element)
                    return@autoCreate
                }

                targets[targetId] = targets[targetId]?.let { it + element } ?: listOf(element)
            },
            onHide = { targetId ->
                targets.remove(targetId)
            }
        )

        if (autoCommitPageView == true && !commitedPageview) {
            engage?.commitPageView()
            commitedPageview = true
        }
    }

    DisposableEffect(Unit) {
        onDispose {
            targets.clear()
            globalElements.clear()
        }
    }

    globalElements.forEach {
        Element(
            element = it,
            modifier = Modifier
                .fillMaxWidth()
                .wrapContentHeight(),
        )
    }
}
