package tech.poool.commons.oak.components

import android.graphics.Color
import android.view.ViewGroup
import android.widget.TextView
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.viewinterop.AndroidView
import androidx.core.text.HtmlCompat
import tech.poool.commons.compose.LocalBlock
import tech.poool.commons.compose.LocalForm
import tech.poool.commons.oak.data.BlockTitleItem
import tech.poool.commons.oak.oakStyleToModifier
import tech.poool.commons.oak.oakToTitleStyle
import tech.poool.commons.oak.renderVariables
import tech.poool.commons.oak.spToPx
import tech.poool.commons.oak.textAlignFromHtml

@Composable
internal fun BlockTitle() {
    val currentBlock = LocalBlock.current
    val block = currentBlock.block as BlockTitleItem
    val formProvider = LocalForm.current

    // This is not enough as AnnotatedString adds a new line at the end of the text
    // when the text is wrapped in a <p> or <div>
    /*Text(
        text = AnnotatedString.fromHtml(block.content),
        style = oakToTitleStyle(block.headingLevel ?: "h1"),
        modifier = Modifier.oakStyleToModifier(
            settings = block.settings,
            styles = block.styles
        ).fillMaxWidth().wrapContentHeight(),
    )*/
    // So we have to use the old TextView to render html and trim it, effectively removing
    // the new line at the end of the text
    val textStyles = oakToTitleStyle(block.headingLevel ?: "h1")
    val finalText = renderVariables(
        HtmlCompat
            .fromHtml(
                "<b>${block.content}</b>",
                HtmlCompat.FROM_HTML_MODE_COMPACT
            ),
        additional = formProvider.getFormTextReplacers()
    )

    AndroidView(
        factory = { context ->
            TextView(context).apply {
                layoutParams = ViewGroup.LayoutParams(
                    ViewGroup.LayoutParams.MATCH_PARENT,
                    ViewGroup.LayoutParams.WRAP_CONTENT
                )
                text = finalText
                setTextColor(Color.BLACK)
                textSize = textStyles.fontSize.value
                if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.P) {
                    lineHeight = spToPx(textStyles.lineHeight.value, context)
                }
                textAlignment = textAlignFromHtml(block.content)
            }
        },
        modifier = Modifier
            .oakStyleToModifier(
                settings = block.settings,
                styles = block.styles,
                palette = currentBlock.palette,
                paletteStyles = currentBlock.paletteStyles,
            )
            .fillMaxWidth()
            .wrapContentHeight(),
        update = { textView ->
            textView.text = finalText
        }
    )
}
