package tech.poool.commons.oak.components

import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Visibility
import androidx.compose.material.icons.filled.VisibilityOff
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.OutlinedTextFieldDefaults
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.input.PasswordVisualTransformation
import androidx.compose.ui.text.input.VisualTransformation
import androidx.compose.ui.unit.dp
import tech.poool.commons.compose.LocalBlock
import tech.poool.commons.compose.LocalForm
import tech.poool.commons.compose.TranslatedText
import tech.poool.commons.helpers.PhoneNumberVisualTransformation
import tech.poool.commons.oak.data.BlockFieldItem
import tech.poool.commons.oak.fromHex
import tech.poool.commons.oak.oakStyleToModifier

@Composable
internal fun BlockTextField() {
    val currentBlock = LocalBlock.current
    val block = currentBlock.block as BlockFieldItem
    val form = LocalForm.current

    var passwordVisible by remember { mutableStateOf(false) }

    val value = form.values[block.fieldKey ?: "unknown"] as String? ?: ""
    val error = form.errors?.find { it.fieldKey == block.fieldKey }

    LaunchedEffect(Unit) {
        if (value.isNotEmpty()) {
            form.validateTextField(block, value)
        }
    }

    Column (modifier = Modifier.fillMaxWidth()) {
        OutlinedTextField(
            value = value,
            enabled = !form.isLoading && !form.isSent,
            modifier = Modifier
                .oakStyleToModifier(
                    settings = block.settings,
                    styles = block.styles,
                    palette = currentBlock.palette,
                    paletteStyles = currentBlock.paletteStyles,
                )
                .background(Color.Transparent)
                .fillMaxWidth(),
            colors = OutlinedTextFieldDefaults.colors().copy(
                unfocusedContainerColor = block.styles?.backgroundColor
                    ?.let { fromHex(it) } ?: MaterialTheme.colorScheme.surface,
            ),
            onValueChange = { form.onTextChange(block, it) },
            label = {
                Text(
                    text = (
                        if (block.fieldLabel.isNullOrEmpty()) block.fieldName
                        else block.fieldLabel
                    ) ?: ""
                )
            },
            placeholder = { Text(text = block.fieldPlaceholder ?: "") },
            singleLine = block.fieldType != "multiline",
            minLines = if (block.fieldType == "multiline") 5 else 1,
            maxLines = if (block.fieldType == "multiline") 10 else 1,
            isError = error != null,
            visualTransformation = (
                when (block.fieldType) {
                    "password" ->
                        if (passwordVisible) VisualTransformation.None
                        else PasswordVisualTransformation()

                    "phone" -> PhoneNumberVisualTransformation()
                    else -> VisualTransformation.None
                }
            ),
            trailingIcon = {
                if (block.fieldType == "password") {
                    IconButton(onClick = { passwordVisible = !passwordVisible }) {
                        Icon(
                            imageVector = if (passwordVisible) Icons.Filled.VisibilityOff
                            else Icons.Filled.Visibility,
                            contentDescription = null
                        )
                    }
                }
            },
            keyboardOptions = KeyboardOptions.Default.copy(
                keyboardType = (
                    when (block.fieldType) {
                        "email" -> KeyboardType.Email
                        "password" -> KeyboardType.Password
                        "phone" -> KeyboardType.Phone
                        else -> KeyboardType.Text
                    }
                )
            ),
        )
        error?.let {
            TranslatedText(
                textKey = it.message ?: "",
                modifier = Modifier.padding(top = 4.dp, bottom = 8.dp),
                style = MaterialTheme.typography.bodySmall,
                color = MaterialTheme.colorScheme.error,
            )
        }
    }
}
