package tech.poool.commons.oak.components

import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.ExposedDropdownMenuBox
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.unit.dp
import tech.poool.commons.compose.LocalBlock
import tech.poool.commons.compose.LocalForm
import tech.poool.commons.compose.TranslatedText
import tech.poool.commons.oak.data.BlockSelectFieldItem
import tech.poool.commons.oak.data.FormFieldOption
import tech.poool.commons.oak.oakStyleToModifier

@OptIn(ExperimentalMaterial3Api::class)
@Composable
internal fun BlockSelectField() {
    val currentBlock = LocalBlock.current
    val block = currentBlock.block as BlockSelectFieldItem
    val form = LocalForm.current

    var value by remember { mutableStateOf("") }
    var expanded by remember { mutableStateOf(false) }

    val error = form.errors?.find { it.fieldKey == block.fieldKey }

    fun onSelect (option: FormFieldOption) {
        expanded = false
        value = option.title
        form.onTextChange(block, option.value)
    }

    Column (modifier = Modifier.fillMaxWidth()) {
        ExposedDropdownMenuBox(
            expanded = expanded,
            onExpandedChange = {
                if (!form.isLoading && !form.isSent) expanded = !expanded
            },
            modifier = Modifier.fillMaxWidth(),
        ) {
            OutlinedTextField(
                readOnly = true,
                value = value,
                onValueChange = {},
                enabled = !form.isLoading && !form.isSent,
                modifier = Modifier
                    .oakStyleToModifier(
                        settings = block.settings,
                        styles = block.styles,
                        palette = currentBlock.palette,
                        paletteStyles = currentBlock.paletteStyles,
                    )
                    .menuAnchor()
                    .fillMaxWidth(),
                placeholder = { Text(text = block.fieldPlaceholder ?: "") },
                label = { Text(text = block.fieldLabel ?: "") },
                isError = error != null,
            )
            ExposedDropdownMenu(expanded = expanded, onDismissRequest = { }) {
                block.fieldOptions.forEach {
                    DropdownMenuItem(
                        text = { Text(text = it.title) },
                        onClick = { onSelect(it) }
                    )
                }
            }
        }
        error?.let {
            TranslatedText(
                textKey = it.message ?: "",
                modifier = Modifier.padding(top = 4.dp, bottom = 8.dp),
                style = MaterialTheme.typography.bodySmall,
                color = MaterialTheme.colorScheme.error,
            )
        }
    }
}
