package tech.poool.commons.oak.components

import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.foundation.layout.FlowColumn
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.ExperimentalLayoutApi
import androidx.compose.foundation.layout.FlowRow
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.layout.wrapContentWidth
import tech.poool.commons.compose.LocalBlock
import tech.poool.commons.oak.data.BlockRowItem
import tech.poool.commons.oak.oakSizeToDp
import tech.poool.commons.oak.oakStyleToModifier
import tech.poool.commons.oak.oakToHorizontalAlignment
import tech.poool.commons.oak.oakToHorizontalArrangement
import tech.poool.commons.oak.oakToVerticalAlignment
import tech.poool.commons.oak.oakToVerticalArrangement

@OptIn(ExperimentalLayoutApi::class)
@Composable
internal fun BlockRow () {
    val currentBlock = LocalBlock.current
    val block = currentBlock.block as BlockRowItem

    val flexDirection = block.settings?.flexDirection
    val reversed = flexDirection?.contains("reverse") == true
    val isColumn = flexDirection?.contains("column") == true

    @Composable
    fun renderBlock(
        autoWeightModifier: Modifier = Modifier,
        flexibleWeightModifier: Modifier = Modifier,
    ) {
        (if (reversed) block.cols.reversed() else block.cols).forEach { item ->
            Block(
                modifier = when (item.size) {
                    "auto" -> autoWeightModifier
                        .then(
                            when (isColumn) {
                                true -> Modifier.height(IntrinsicSize.Min).wrapContentHeight()
                                else -> Modifier.width(IntrinsicSize.Min).wrapContentWidth()
                            }
                        )
                    "flexible", "fluid" -> flexibleWeightModifier
                    null -> flexibleWeightModifier
                    else -> Modifier.fillMaxWidth((item.size ?: "12").toFloat() / 12)
                },
                block = item,
                palette = currentBlock.palette,
                paletteStyles = currentBlock.paletteStyles,
                additionalTypes = currentBlock.additionalTypes,
                onDestroy = currentBlock.onDestroy,
                onClick = currentBlock.onClick,
                onTriggerEvent = currentBlock.onTriggerEvent,
            )
        }
    }

    if (isColumn) {
        FlowColumn (
            modifier = currentBlock.modifier
                .fillMaxWidth()
                .oakStyleToModifier(
                    settings = block.settings,
                    styles = block.styles,
                    palette = currentBlock.palette,
                    paletteStyles = currentBlock.paletteStyles,
                ),
            // alignItems (vertical) & justifyContent (horizontal) are reversed on col direction
            // alignItems becomes horizontal, justifyContent becomes vertical
            horizontalArrangement = Arrangement
                .oakToHorizontalArrangement(block.settings?.alignItems),
            verticalArrangement = when (block.settings?.justifyContent) {
                "space-between", "space-around", "space-evenly" ->
                    Arrangement.oakToVerticalArrangement(block.settings.justifyContent)
                else ->
                    Arrangement.spacedBy(
                        oakSizeToDp(block.settings?.gap ?: "0"),
                        oakToVerticalAlignment(block.settings?.justifyContent)
                    )
            },
        ) {
            renderBlock(
                autoWeightModifier = Modifier
                    .align(oakToHorizontalAlignment(block.settings?.alignItems)),
                flexibleWeightModifier = Modifier
                    .weight(1f)
                    .align(oakToHorizontalAlignment(block.settings?.alignItems)),
            )
        }

        return
    }

    FlowRow(
        modifier = currentBlock.modifier
            .fillMaxWidth()
            .oakStyleToModifier(
                settings = block.settings,
                styles = block.styles,
                palette = currentBlock.palette,
                paletteStyles = currentBlock.paletteStyles,
            ),
        horizontalArrangement = when (block.settings?.justifyContent) {
            "space-between", "space-around", "space-evenly" ->
                Arrangement.oakToHorizontalArrangement(block.settings.justifyContent)
            else ->
                Arrangement.spacedBy(
                    oakSizeToDp(block.settings?.gap ?: "0"),
                    oakToHorizontalAlignment(block.settings?.justifyContent)
                )
        },
        verticalArrangement = Arrangement.oakToVerticalArrangement(block.settings?.alignItems),
    ) {
        renderBlock(
            autoWeightModifier = Modifier
                .align(oakToVerticalAlignment(block.settings?.alignItems)),
            flexibleWeightModifier = Modifier
                .weight(1f)
                .align(oakToVerticalAlignment(block.settings?.alignItems)),
        )
    }
}
