package tech.poool.commons.oak.components

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.selection.selectable
import androidx.compose.foundation.selection.selectableGroup
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.RadioButton
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.semantics.Role
import androidx.compose.ui.unit.dp
import tech.poool.commons.compose.LocalBlock
import tech.poool.commons.compose.LocalForm
import tech.poool.commons.compose.TranslatedText
import tech.poool.commons.oak.data.BlockRadioFieldItem
import tech.poool.commons.oak.data.FormFieldOption
import tech.poool.commons.oak.oakStyleToModifier

@Composable
internal fun BlockRadioField() {
    val currentBlock = LocalBlock.current
    val block = currentBlock.block as BlockRadioFieldItem
    val form = LocalForm.current

    val error = form.errors?.find { it.fieldKey == block.fieldKey }
    val value = form.values[block.fieldKey ?: "unknown"] as String? ?: ""

    fun onSelect (option: FormFieldOption) {
        form.onTextChange(block, option.value)
    }

    Column (
        modifier = Modifier
            .oakStyleToModifier(
                settings = block.settings,
                styles = block.styles,
                palette = currentBlock.palette,
                paletteStyles = currentBlock.paletteStyles,
            )
            .fillMaxWidth()
            .selectableGroup()
    ) {
        block.fieldOptions.forEach {
            Row (
                modifier = Modifier
                    .fillMaxWidth()
                    .selectable(
                        selected = value == it.value,
                        onClick = { onSelect(it) },
                        role = Role.RadioButton,
                    )
                    .padding(vertical = 8.dp),
                verticalAlignment = Alignment.CenterVertically,
                horizontalArrangement = Arrangement.spacedBy(4.dp),
            ) {
                RadioButton(
                    selected = value == it.value,
                    onClick = null,
                    modifier = Modifier.padding(0.dp),
                    enabled = !form.isLoading && !form.isSent,
                )
                Text(
                    text = it.title,
                )
            }
        }

        error?.let {
            TranslatedText(
                textKey = it.message ?: "",
                modifier = Modifier.padding(top = 4.dp, bottom = 8.dp),
                style = MaterialTheme.typography.bodySmall,
                color = MaterialTheme.colorScheme.error,
            )
        }
    }
}
