package tech.poool.commons.oak.components

import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.ui.Modifier
import tech.poool.commons.compose.LocalBlock
import tech.poool.commons.compose.LocalCommons
import tech.poool.commons.oak.data.*

@Composable
internal fun Block (
    modifier: Modifier = Modifier,
    block: BlockItem,
    additionalTypes: Map<String, @Composable (BlockItem) -> Unit> = emptyMap(),
    palette: ColorPalette? = null,
    paletteStyles: Map<String, Any?>? = null,
    textReplacers: Map<String, String> = emptyMap(),
    onClick: ((eventData: Map<String, Any?>?) -> Unit)? = null,
    onTriggerEvent: ((eventType: String, eventData: Map<String, Any?>?) -> Unit)? = null,
    onDestroy: ((buttonId: String?) -> Unit)? = null,
){
    val commons = LocalCommons.current

    if (
        (commons.isReleased && block.settings?.showAfterRelease == false) ||
        (!commons.isReleased && block.settings?.showBeforeRelease == false)
    ) {
        return
    }

    ShouldDisplay(
        block = block,
        textReplacers = textReplacers,
    ) {
        Field(block) {
            CompositionLocalProvider(
                LocalBlock provides BlockProviderData(
                    block = block,
                    modifier = modifier,
                    palette = palette,
                    paletteStyles = paletteStyles,
                    additionalTypes = additionalTypes,
                    onClick = onClick,
                    onTriggerEvent = onTriggerEvent,
                    onDestroy = onDestroy,
                ),
            ) {
                when (block) {
                    is BlockRowItem -> BlockRow()
                    is BlockColItem -> BlockCol()
                    is BlockTitleItem -> BlockTitle()
                    is BlockTextItem -> BlockText()
                    is BlockButtonItem -> BlockButton()
                    is BlockImageItem -> BlockImage()
                    is BlockEmptySpaceItem -> BlockEmptySpace()
                    is BlockListItem -> BlockList()
                    is BlockFoldableItem -> BlockFoldable()
                    is BlockClickableItem -> BlockClickable()
                    is BlockTextFieldItem,
                    is BlockEmailFieldItem,
                    is BlockPasswordFieldItem,
                    is BlockPhoneFieldItem,
                    is BlockMultilineFieldItem -> BlockTextField()
                    is BlockSelectFieldItem -> BlockSelectField()
                    is BlockRadioFieldItem -> BlockRadioField()
                    is BlockCheckboxFieldItem -> BlockCheckboxField()
                    is BlockDateFieldItem -> BlockDateField()
                    is BlockHiddenFieldItem -> BlockHiddenField()
                    else -> {
                        if (additionalTypes[block.type] != null) {
                            additionalTypes[block.type]?.invoke(block)
                        } else {
                            commons.logger?.w("Unknown block type: ${block.type}")
                        }
                    }
                }
            }
        }
    }
}

internal data class BlockProviderData(
    val block: BlockItem? = null,
    val modifier: Modifier = Modifier,
    val palette: ColorPalette? = null,
    val paletteStyles: Map<String, Any?>? = null,
    val additionalTypes: Map<String, @Composable (BlockItem) -> Unit> = emptyMap(),
    val onClick: ((eventData: Map<String, Any?>?) -> Unit)? = null,
    val onTriggerEvent: ((eventType: String, eventData: Map<String, Any?>?) -> Unit)? = null,
    val onDestroy: ((buttonId: String?) -> Unit)? = null,
)
