package tech.poool.commons.network

import java.util.concurrent.TimeUnit
import kotlinx.coroutines.suspendCancellableCoroutine
import okhttp3.Interceptor
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.Response
import okhttp3.logging.HttpLoggingInterceptor
import tech.poool.commons.config.BaseConfig
import tech.poool.commons.config.get
import tech.poool.commons.logger.BaseLogger
import tech.poool.commons.network.utils.CookieManager
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException

internal open class BaseNetwork(
    private val config: BaseConfig,
    private val logger: BaseLogger,
    isDebug: Boolean = false,
) {
    private val okHttpBuilder: OkHttpClient.Builder
    private var interceptors: MutableList<Interceptor> = mutableListOf()
    private val cookieManager by lazy {
        CookieManager(isCookieEnabled(), logger)
    }

    private fun isCookieEnabled(): Boolean {
        return this.config.get<Boolean?>("cookiesEnabled") ?: false
    }

    init {
        val builder = OkHttpClient.Builder()
            .connectTimeout(10, TimeUnit.SECONDS)
            .readTimeout(10, TimeUnit.SECONDS)
            .retryOnConnectionFailure(true)

        if (isDebug) {
            val logger = HttpLoggingInterceptor()
                .setLevel(HttpLoggingInterceptor.Level.BASIC)

            interceptors.add(logger)
        }

        interceptors.forEach { builder.addInterceptor(it) }

        okHttpBuilder = builder
    }

    suspend fun request(request: Request): Response? {
        isCookieEnabled().let {
            if (it) okHttpBuilder.cookieJar(cookieManager)
        }
        val okHttp = okHttpBuilder.build()

        return suspendCancellableCoroutine { continuation ->
            continuation.invokeOnCancellation { okHttp.dispatcher.cancelAll() }

            okHttp.newCall(request).enqueue(object : okhttp3.Callback {
                override fun onFailure(call: okhttp3.Call, e: java.io.IOException) {
                    continuation.resumeWithException(e)
                }

                override fun onResponse(call: okhttp3.Call, response: Response) {
                    continuation.resume(response)
                }
            })
        }
    }

    companion object {
        const val SCOPE = "NETWORK"
    }
}
