package tech.poool.commons.config

import tech.poool.commons.logger.BaseLogger

internal open class BaseConfig (
    val baseData: Map<String, Any?> = mapOf(
        "appId" to null,
    ),
    private val flushable: List<String> = listOf(),
) {
    var logger: BaseLogger? = null
        private set

    val readOnlies: MutableList<String> = mutableListOf()
    val customData: MutableMap<String, Any?> = mutableMapOf()

    fun setLogger(logger: BaseLogger) {
        this.logger = logger
    }

    private fun isInternalKey (key: String): Boolean {
        return key.matches("^_.*".toRegex())
    }

    fun getAll(): Map<String, Any?> {
        return baseData + customData
    }

    fun <T>setVal (key: String, value: T?, readOnly: Boolean? = false, isInternal: Boolean? = false): Boolean {
        val camelCaseKey = normalizeKey(key)

        if (
            value == null ||
            (isInternalKey(key) && isInternal == false) ||
            (value is String && value.trim().isEmpty()) ||
            readOnlies.contains(camelCaseKey)
        ) {
            return false
        }

        if (baseData.keys.contains(camelCaseKey)) {
//            TODO: throw exception if value type is wrong

            customData[camelCaseKey] = value

            if ((readOnly != null) && readOnly) {
                readOnlies.add(camelCaseKey)
            }

            return true
        }

        return false
    }

    fun reset() {
        readOnlies.clear()

        baseData.forEach {
            if (it.key in flushable) {
                customData.remove(it.key)
            }
        }
    }

    fun normalizeKey(name: String): String {
        if (isInternalKey(name)) {
            return name
        }

        return name
            .replace("([-_][a-z])".toRegex(RegexOption.IGNORE_CASE)) { it.value.uppercase() }
            .replace("-", "")
            .replace("_", "")
    }

    companion object {
        const val SCOPE= "CONFIG"
    }
}

internal inline fun <reified T> BaseConfig.get(key: String): T {
    return try {
        (customData[normalizeKey(key)] ?: baseData[normalizeKey(key)]) as T
    } catch (e: Exception) {
        logger?.e(
            "You cannot set a value of type" +
                    " ${customData[normalizeKey(key)]?.let { it::class.java }} to <$key> key",
            BaseConfig.SCOPE
        )

        baseData[normalizeKey(key)] as T
    }
}
