package tech.poool.engage.core

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.Deferred
import kotlinx.coroutines.async
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import tech.poool.engage.EngageConfig
import tech.poool.engage.EngageDb
import tech.poool.engage.EngageEventsManager
import tech.poool.engage.EngageLogger
import tech.poool.engage.network.responses.ElementResponse
import tech.poool.engage.network.responses.AutoCreateResponse
import tech.poool.engage.network.responses.SentResponse
import tech.poool.engage.network.responses.SuccessResponse
import tech.poool.engage.EngageTexts
import tech.poool.engage.EngageVariables
import tech.poool.engage.network.EngageNetwork

internal class EngageViewModel(internal val dataConfig: EngageDataConfig) : ViewModel() {
    private val globalExceptionHandler = CoroutineExceptionHandler { _, throwable ->
        dataConfig.logger.e(throwable.message ?: throwable.toString())
    }

    private val _filters = MutableStateFlow<List<String>?>(null)
    val filters: StateFlow<List<String>?> = _filters.asStateFlow()

    fun updateFilters(filters: List<String>?) {
        _filters.value = filters
    }

    suspend fun getElement(slug: String): Deferred<ElementResponse?> {
        return viewModelScope.async {
            dataConfig.api.getElement(slug)
        }
    }

    suspend fun autoCreate(filters: List<String>? = null): Deferred<AutoCreateResponse?> {
        return viewModelScope.async {
            dataConfig.api.autoCreate(filters)
        }
    }

    suspend fun sendEvent (data: Map<String, *>): Deferred<SuccessResponse?> {
        return viewModelScope.async(globalExceptionHandler) {
            dataConfig.api.sendEvent(data)
        }
    }

    suspend fun submitForm (data: Map<String, Any>): Deferred<SentResponse?> {
        return viewModelScope.async(globalExceptionHandler) {
            dataConfig.api.submitForm(data)
        }
    }
}

internal data class EngageDataConfig(
    var config: EngageConfig,
    var variables: EngageVariables,
    var texts: EngageTexts,
    var events: EngageEventsManager,
    var logger: EngageLogger,
    var api: EngageNetwork,
    var db: EngageDb?,
)
