package tech.poool.commons.oak.components

import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.DatePicker
import androidx.compose.material3.DatePickerDefaults
import androidx.compose.material3.DatePickerDialog
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.OutlinedTextFieldDefaults
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.rememberDatePickerState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.onFocusChanged
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalFocusManager
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import tech.poool.commons.compose.LocalBlock
import tech.poool.commons.compose.LocalForm
import tech.poool.commons.compose.TranslatedText
import tech.poool.commons.oak.data.BlockDateFieldItem
import tech.poool.commons.oak.fromHex
import tech.poool.commons.oak.millisToDate
import tech.poool.commons.oak.oakStyleToModifier
import tech.poool.commons.oak.selectableDates

@OptIn(ExperimentalMaterial3Api::class)
@Composable
internal fun BlockDateField() {
    val currentBlock = LocalBlock.current
    val block = currentBlock.block as BlockDateFieldItem
    val form = LocalForm.current

    val value = form.values[block.fieldKey ?: "unknown"] as String? ?: ""
    val error = form.errors?.find { it.fieldKey == block.fieldKey }

    val focusManager = LocalFocusManager.current
    var showDatePicker by remember { mutableStateOf(false) }
    val datePickerState = rememberDatePickerState(
        selectableDates = selectableDates(block.minDate, block.maxDate) ?:
            DatePickerDefaults.AllDates
    )

    fun closeDatePicker() {
        showDatePicker = false
        focusManager.clearFocus()
    }

    fun confirmDatePicker() {
        if (datePickerState.selectedDateMillis != null) {
            val date = millisToDate(
                datePickerState.selectedDateMillis!!,
                form.dateFormat ?: "dd/MM/yyyy"
            )

            form.onTextChange(block, date)
        }

        closeDatePicker()
    }

    Column (modifier = Modifier.fillMaxWidth()) {
        OutlinedTextField(
            modifier = Modifier
                .onFocusChanged { focusState ->
                    if (focusState.isFocused) {
                        showDatePicker = true
                    }
                }
                .oakStyleToModifier(
                    settings = block.settings,
                    styles = block.styles,
                    palette = currentBlock.palette,
                    paletteStyles = currentBlock.paletteStyles,
                )
                .background(Color.Transparent)
                .fillMaxWidth(),
            value = value,
            onValueChange = {},
            label = {
                Text(
                    text = (
                        if (block.fieldLabel.isNullOrEmpty()) block.fieldName
                        else block.fieldLabel
                    ) ?: ""
                )
            },
            placeholder = { Text(text = block.fieldPlaceholder ?: "") },
            enabled = !form.isLoading && !form.isSent,
            isError = error != null,
            colors = OutlinedTextFieldDefaults.colors().copy(
                unfocusedContainerColor = block.styles?.backgroundColor
                    ?.let { fromHex(it) } ?: MaterialTheme.colorScheme.surface,
            ),
            readOnly = true,
        )

        if (showDatePicker) {
            DatePickerDialog(
                onDismissRequest = {
                    closeDatePicker()
                },
                confirmButton = {
                    TextButton(onClick = { confirmDatePicker() }) {
                        Text(text = stringResource(android.R.string.ok))
                    }
                },
                dismissButton = {
                    TextButton(onClick = { closeDatePicker() }) {
                        Text(text = stringResource(android.R.string.cancel))
                    }
                },
            ) {
                DatePicker(
                    state = datePickerState,
                    showModeToggle = false,
                )
            }
        }

        error?.let {
            TranslatedText(
                textKey = it.message ?: "",
                modifier = Modifier.padding(top = 4.dp, bottom = 8.dp),
                style = MaterialTheme.typography.bodySmall,
                color = MaterialTheme.colorScheme.error,
            )
        }
    }
}
