package tech.poool.commons.oak.components

import java.net.URLEncoder
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import tech.poool.commons.events.BaseEvents
import tech.poool.commons.compose.LocalBlock
import tech.poool.commons.compose.LocalCommons
import tech.poool.commons.config.get
import tech.poool.commons.oak.data.BlockClickableItem
import tech.poool.commons.oak.oakStyleToModifier

@Composable
internal fun BlockClickable() {
    val currentBlock = LocalBlock.current
    val commons = LocalCommons.current
    val commonsProvider = LocalCommons.current

    val block = currentBlock.block as BlockClickableItem
    val buttonId = block.customId

    fun onClick () {
        currentBlock.onClick?.invoke(mapOf(
            "type" to "click",
            // This should be the current page url, but we don't have that on mobile
            "url" to (block.url ?: ""),
            "target" to mapOf(
                "type" to block.type,
                "id" to buttonId,
            ),
        ))

        when (block.action) {
            "event" -> {
                currentBlock.onTriggerEvent?.invoke(BaseEvents.CLICK, mapOf(
                    "name" to block.event,
                    "url" to block.url,
                    "buttonId" to block.customId,
                ))
            }
            "destroy" -> {
                currentBlock.onDestroy?.invoke(buttonId)
            }
            else -> {
                val returnUrl = commons.config?.get<String?>("customReturnUrl")
                val url = block.url
                    ?.replace("{returnUrl}", URLEncoder.encode(returnUrl ?: "", "UTF-8"))
                    ?: ""

                if (url.isNotEmpty()) {
                    commonsProvider.openUri(url)
                }
            }
        }
    }

    Column(
        modifier = Modifier
            .oakStyleToModifier(
                settings = block.settings,
                styles = block.styles,
                palette = currentBlock.palette,
                paletteStyles = currentBlock.paletteStyles,
            )
            .clickable { onClick() }
            .fillMaxWidth()
    ) {
        block.content.forEach { item ->
            Block(
                block = item,
                palette = currentBlock.palette,
                paletteStyles = currentBlock.paletteStyles,
                additionalTypes = currentBlock.additionalTypes,
                onDestroy = currentBlock.onDestroy,
                onClick = currentBlock.onClick,
                onTriggerEvent = currentBlock.onTriggerEvent,
            )
        }
    }
}
