package tech.poool.commons.oak.components

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.selection.toggleable
import androidx.compose.material3.Checkbox
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.LocalMinimumInteractiveComponentEnforcement
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.semantics.Role
import androidx.compose.ui.unit.dp
import tech.poool.commons.compose.LocalBlock
import tech.poool.commons.compose.LocalForm
import tech.poool.commons.compose.TranslatedText
import tech.poool.commons.oak.data.BlockCheckboxFieldItem
import tech.poool.commons.oak.oakStyleToModifier

@OptIn(ExperimentalMaterial3Api::class)
@Composable
internal fun BlockCheckboxField() {
    val currentBlock = LocalBlock.current
    val block = currentBlock.block as BlockCheckboxFieldItem
    val form = LocalForm.current

    val error = form.errors?.find { it.fieldKey == block.fieldKey }
    val value = form.values[block.fieldKey ?: "unknown"] as Boolean? ?: false

    fun onToggle (checked: Boolean) {
        form.onToggle(block, checked)
    }

    Column (
        modifier = Modifier
            .oakStyleToModifier(
                settings = block.settings,
                styles = block.styles,
                palette = currentBlock.palette,
                paletteStyles = currentBlock.paletteStyles,
            )
            .fillMaxWidth()
            .padding(0.dp),
    ) {
        Row (
            modifier = Modifier
                .fillMaxWidth()
                .toggleable(
                    value = value,
                    onValueChange = { onToggle(it) },
                    role = Role.Checkbox,
                )
                .padding(vertical = 8.dp, horizontal = 0.dp),
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = Arrangement.spacedBy(4.dp),
        ) {
            CompositionLocalProvider(LocalMinimumInteractiveComponentEnforcement provides false) {
                Checkbox(
                    checked = value,
                    onCheckedChange = { onToggle(it) },
                    modifier = Modifier.padding(0.dp),
                    enabled = !form.isLoading && !form.isSent,
                )
            }
            Text(
                text = (
                    if (block.fieldLabel.isNullOrEmpty()) block.fieldName
                    else block.fieldLabel
                ) ?: ""
            )
        }

        error?.let {
            TranslatedText(
                textKey = it.message ?: "",
                modifier = Modifier.padding(top = 4.dp, bottom = 8.dp),
                style = MaterialTheme.typography.bodySmall,
                color = MaterialTheme.colorScheme.error,
            )
        }
    }
}
