package tech.poool.commons.oak.components

import java.net.URLEncoder
import android.widget.TextView
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.Button
import androidx.compose.material3.ButtonDefaults
import androidx.compose.material3.MaterialTheme
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.unit.dp
import androidx.compose.ui.viewinterop.AndroidView
import androidx.core.text.HtmlCompat
import tech.poool.commons.events.BaseEvents
import tech.poool.commons.compose.LocalBlock
import tech.poool.commons.compose.LocalCommons
import tech.poool.commons.compose.LocalForm
import tech.poool.commons.config.get
import tech.poool.commons.oak.data.BlockButtonItem
import tech.poool.commons.oak.fromHex
import tech.poool.commons.oak.oakStyleToModifier
import tech.poool.commons.oak.renderVariables
import tech.poool.commons.oak.spToPx

@Composable
internal fun BlockButton() {
    val currentBlock = LocalBlock.current
    val commons = LocalCommons.current
    val form = LocalForm.current

    val block = currentBlock.block as BlockButtonItem
    val buttonId = block.customId

    val finalText = renderVariables(
        HtmlCompat
            .fromHtml(block.content, HtmlCompat.FROM_HTML_MODE_COMPACT)
            .trim()
            .toString(),
        additional = form.getFormTextReplacers()
    )

    val setUpChangeValue = if (block.action == "change-field" && block.fieldOperation == "set") {
        renderVariables(block.fieldValue ?: "", additional = form.getFormTextReplacers())
    } else {
        ""
    }

    fun onClick () {
        currentBlock.onClick?.invoke(mapOf(
            "type" to "click",
            // This should be the current page url, but we don't have that on mobile
            "url" to (block.url ?: ""),
            "target" to mapOf(
                "type" to block.type,
                "id" to buttonId,
            ),
        ))

        when (block.action) {
            "event" -> {
                currentBlock.onTriggerEvent?.invoke(BaseEvents.CLICK, mapOf(
                    "name" to block.event,
                    "url" to block.url,
                    "id" to block.customId,
                    "type" to block.type,
                ))
            }
            "submit" -> {
                form.onSubmit()
            }
            "destroy" -> {
                currentBlock.onDestroy?.invoke(block.customId)
            }
            "change-field" -> {
                val field = form.fields?.find { it.fieldKey == block.fieldKey }

                if (field == null) {
                    commons.logger?.w("Field <${block.fieldKey}> not found, skipping")

                    return
                }

                try {
                    when (block.fieldOperation) {
                        "inc" -> {
                            form.onTextChange(
                                field,
                                (form.getFieldValue(field).toInt() +
                                        (block.fieldValue?.toInt() ?: 1)).toString()
                            )
                        }
                        "dec" -> {
                            form.onTextChange(
                                field,
                                (form.getFieldValue(field).toInt() -
                                        (block.fieldValue?.toInt() ?: 1)).toString()
                            )
                        }
                        else -> {
                            form.onTextChange(field, setUpChangeValue)
                        }
                    }
                } catch (e: Exception) {
                    commons.logger?.w("Field <${block.fieldKey}> is not a number, skipping")
                    commons.logger?.e(e.toString())
                }
            }
            else -> {
                val returnUrl = commons.config?.get<String?>("customReturnUrl")
                val url = block.url
                    ?.replace("{returnUrl}", URLEncoder.encode(returnUrl ?: "", "UTF-8"))
                    ?: ""

                if (url.isNotEmpty()) {
                    commons.openUri(url)
                }
            }
        }
    }

    Column(
        horizontalAlignment = Alignment.CenterHorizontally,
        modifier = Modifier
            .oakStyleToModifier(
                settings = block.settings,
                styles = block.styles,
                palette = currentBlock.palette,
                paletteStyles = currentBlock.paletteStyles,
            )
            .background(Color.Transparent)
            .fillMaxWidth()
    ) {
        Button(
            onClick = { onClick() },
            shape = RoundedCornerShape(4.dp),
            enabled = block.action != "submit" || !form.isLoading && !form.isSent,
            colors = ButtonDefaults.buttonColors(
                containerColor =
                    block.styles?.backgroundColor?.let { if (it.isNotEmpty()) fromHex(it) else null }
                    ?: Color.Black,
            )
        ) {
            // This is not enough as AnnotatedString adds a new line at the end of the text
            // when the text is wrapped in a <p> or <div>
            /*Text(
                text = AnnotatedString.fromHtml(block.content.trim()),
            )*/
            // So we have to use the old TextView to render html and trim it, effectively removing
            // the new line at the end of the text
            val textStyles = MaterialTheme.typography.bodyLarge
            AndroidView(
                factory = { context ->
                    TextView(context).apply {
                        text = finalText
                        textSize = textStyles.fontSize.value
                        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.P) {
                            lineHeight = spToPx(textStyles.lineHeight.value, context)
                        }
                        setTextColor(0xFFFFFFFF.toInt())
                    }
                },
                update = {
                    it.text = finalText
                }
            )
        }
    }
}
