package tech.poool.commons.oak

import android.graphics.Bitmap
import android.graphics.BlurMaskFilter
import android.graphics.drawable.BitmapDrawable
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.MaterialTheme
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.draw.drawBehind
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.ImageBitmap
import androidx.compose.ui.graphics.ImageShader
import androidx.compose.ui.graphics.Paint
import androidx.compose.ui.graphics.RectangleShape
import androidx.compose.ui.graphics.ShaderBrush
import androidx.compose.ui.graphics.asAndroidBitmap
import androidx.compose.ui.graphics.asImageBitmap
import androidx.compose.ui.graphics.drawscope.drawIntoCanvas
import androidx.compose.ui.graphics.drawscope.withTransform
import androidx.compose.ui.graphics.nativeCanvas
import androidx.compose.ui.graphics.toArgb
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import coil.ImageLoader
import coil.decode.SvgDecoder
import coil.request.ImageRequest
import tech.poool.commons.oak.data.BlockSettings
import tech.poool.commons.oak.data.BlockStyles
import tech.poool.commons.oak.data.ColorPalette
import tech.poool.commons.oak.data.ColorPaletteDefinitionColor

@Composable
internal fun Modifier.oakStyleToModifier(
    settings: BlockSettings? = null,
    styles: BlockStyles? = null,
    skip: List<String> = emptyList(),
    palette: ColorPalette? = null,
    paletteStyles: Map<String, Any?>? = null,
): Modifier {
    if (settings == null && styles == null) return this then Modifier

    return this then Modifier
        .then(settings?.let { setting ->
            Modifier
                .then(
                    setting.height?.let {
                        if (!skip.contains("height")) Modifier.height(oakSizeToDp(it))
                        else Modifier
                    } ?: Modifier
                )
                .then(
                    setting.width?.let {
                        if (!skip.contains("width")) Modifier.width(oakSizeToDp(it))
                        else Modifier
                    } ?: Modifier
                )
        } ?: Modifier)
        .then(styles?.let {
            Modifier
                .padding(
                    // Margin
                    top = oakSizeToDp(it.marginTop),
                    start = oakSizeToDp(it.marginLeft),
                    end = oakSizeToDp(it.marginRight),
                    bottom = oakSizeToDp(it.marginBottom),
                )
        } ?: Modifier)
        .then(styles?.let {
            if (hasShadow(styles))
                Modifier
                    .oakShadow(
                        offsetX = oakSizeToDp(it.boxShadowX),
                        offsetY = oakSizeToDp(it.boxShadowY),
                        blurRadius = oakSizeToDp(it.boxShadowBlur),
                        spread = oakSizeToDp(it.boxShadowSpread),
                        color = oakColorToColor(it.boxShadowColor ?: "#000", palette = palette, paletteStyles = paletteStyles),
                    )
            else
                Modifier
        } ?: Modifier)
        .then(styles?.let {
            if (hasBorderRadius(styles) && !hasBorder(styles))
                Modifier
                    .clip(RoundedCornerShape(
                        topStart = oakSizeToDp(it.borderTopLeftRadius),
                        topEnd = oakSizeToDp(it.borderTopRightRadius),
                        bottomEnd = oakSizeToDp(it.borderBottomRightRadius),
                        bottomStart = oakSizeToDp(it.borderBottomLeftRadius),
                    ))
            else
                Modifier
        } ?: Modifier)
        .then(styles?.let {
            if (hasBorder(styles))
                Modifier
                    .border(
                        width = oakSizeToDp(it.borderWidth),
                        color = oakColorToColor(it.borderColor ?: "#000", palette = palette, paletteStyles = paletteStyles),
                        shape =
                            if (hasBorderRadius(styles))
                                    RoundedCornerShape(
                                        topStart = oakSizeToDp(it.borderTopLeftRadius),
                                        topEnd = oakSizeToDp(it.borderTopRightRadius),
                                        bottomEnd = oakSizeToDp(it.borderBottomRightRadius),
                                        bottomStart = oakSizeToDp(it.borderBottomLeftRadius),
                                    )
                            else
                                RectangleShape
                    )
            else
                Modifier
        } ?: Modifier)
        .then(styles?.let {
            Modifier
                .background(
                    color = it.backgroundColor?.let { color ->
                        oakColorToColor(color, palette = palette, paletteStyles = paletteStyles)
                    } ?: Color.Transparent,
                )
                .oakBackgroundImage(styles)
                .padding( // Padding
                    top = oakSizeToDp(it.paddingTop),
                    start = oakSizeToDp(it.paddingLeft),
                    end = oakSizeToDp(it.paddingRight),
                    bottom = oakSizeToDp(it.paddingBottom),
                )
        } ?: Modifier)
}

@Composable
internal fun Modifier.oakBackgroundImage(
    styles: BlockStyles? = null,
): Modifier {
    val context = LocalContext.current

    var brush by remember { mutableStateOf<ShaderBrush?>(null) }
    var imageBitmap by remember { mutableStateOf<ImageBitmap?>(null) }

    LaunchedEffect(Unit) {
        if (styles?.backgroundImage == null) return@LaunchedEffect

        val imageLoader = ImageLoader.Builder(context)
            .components { add(SvgDecoder.Factory()) }
            .build()
        val request = ImageRequest.Builder(context)
            .data(styles.backgroundImage.url)
            .size(coil.size.Size.ORIGINAL)
            .build()
        val result = imageLoader.execute(request)
        val bitmap = (result.drawable as BitmapDrawable).bitmap
        imageBitmap = bitmap.asImageBitmap()

        brush =
            ShaderBrush(
                shader = ImageShader(
                    image = bitmap.asImageBitmap(),
                    tileModeX = oakToTileModeX(styles),
                    tileModeY = oakToTileModeY(styles),
                ),
            )
    }

    if (imageBitmap == null) return this

    return this.drawBehind {
        val imageSize = oakToBackgroundSize(
            imageSize = Size(imageBitmap!!.width.toFloat(), imageBitmap!!.height.toFloat()),
            parentSize = this.size,
            backgroundSize = styles?.backgroundSize ?: "cover"
        )
        val imageOffset = oakToBackgroundPosition(
            imageSize = imageSize,
            parentSize = this.size,
            position = styles?.backgroundPosition ?: "center"
        )

        withTransform({
            translate(left = imageOffset.x, top = imageOffset.y)
        }) {
            drawIntoCanvas {
                it.nativeCanvas.drawPaint(Paint().asFrameworkPaint().apply {
                    shader = ImageShader(
                        image = Bitmap
                            .createScaledBitmap(
                                imageBitmap!!.asAndroidBitmap(),
                                imageSize.width.toInt(),
                                imageSize.height.toInt(),
                                true
                            )
                            .asImageBitmap(),
                        tileModeX = oakToTileModeX(styles),
                        tileModeY = oakToTileModeY(styles),
                    )
                })
            }
        }
    }
}

internal fun Arrangement.oakToHorizontalArrangement(
    justifyContent: String? = null
): Arrangement.Horizontal {
    return when (justifyContent) {
        "flex-start" -> Start
        "flex-end" -> End
        "center" -> Center
        "space-between" -> SpaceBetween
        "space-around" -> SpaceAround
        "space-evenly" -> SpaceEvenly
        else -> Start
    }
}

internal fun Arrangement.oakToVerticalArrangement(
    alignItems: String? = null
): Arrangement.Vertical {
    return when (alignItems) {
        "flex-start" -> Top
        "flex-end" -> Bottom
        "center" -> Center
        "space-between" -> SpaceBetween
        "space-around" -> SpaceAround
        "space-evenly" -> SpaceEvenly
        else -> Top
    }
}

internal fun oakToHorizontalAlignment(
    justifyContent: String? = null
): Alignment.Horizontal {
    return when (justifyContent) {
        "flex-start" -> Alignment.Start
        "flex-end" -> Alignment.End
        "center" -> Alignment.CenterHorizontally
        else -> Alignment.Start
    }
}

internal fun oakToVerticalAlignment(
    alignItems: String? = null
): Alignment.Vertical {
    return when (alignItems) {
        "flex-start" -> Alignment.Top
        "flex-end" -> Alignment.Bottom
        "center" -> Alignment.CenterVertically
        else -> Alignment.Top
    }
}

@Composable
internal fun oakToTitleStyle(
    headingLevel: String?
): TextStyle {
    return when (headingLevel) {
        "h1" -> MaterialTheme.typography.headlineLarge
        "h2" -> MaterialTheme.typography.headlineMedium
        "h3" -> MaterialTheme.typography.headlineSmall
        "h4" -> MaterialTheme.typography.titleLarge
        "h5" -> MaterialTheme.typography.titleMedium
        "h6" -> MaterialTheme.typography.titleSmall
        else -> MaterialTheme.typography.bodyMedium
    }
}

internal fun oakToImageAlignment (
    settings: BlockSettings? = null
): Alignment {
    return when (settings?.textAlign) {
        "left" -> Alignment.CenterStart
        "right" -> Alignment.CenterEnd
        "center" -> Alignment.Center
        else -> Alignment.CenterStart
    }
}

internal fun oakToTextAlignment (
    settings: BlockSettings? = null
): TextAlign {
    return when (settings?.textAlign ?: settings?.alignment) {
        "left" -> TextAlign.Start
        "right" -> TextAlign.End
        "center" -> TextAlign.Center
        else -> TextAlign.Start
    }
}

internal fun oakColorToColor(
    color: String,
    palette: ColorPalette? = null,
    paletteStyles: Map<String, Any?>? = null
): Color {
    try {
        if (color.startsWith("var(--")) {
            // ex: var(--skin-background-color)
            val name = color
                // 10 = var(--skin-, length - 1 = ) -> background-color
                .substring(10, color.length - 1)
                // background-color -> background, color
                .split("-")
                // background, color -> BackgroundColor
                .joinToString("") { it.replaceFirstChar { c -> c.uppercase() } }
                // BackgroundColor -> backgroundColor
                .replaceFirstChar { it.lowercase() }

            when (val style = paletteStyles?.get(name)) {
                is String ->
                    return fromHex(style)
                is Map<*, *> -> {
                    val value = palette?.styles?.find {
                        it.type == "color" && it.key == style["key"]
                    } as ColorPaletteDefinitionColor?

                    if (value?.value != null) {
                        return fromHex(value.value)
                    }
                }
                else -> {
                    return Color.Transparent
                }
            }
        } else {
            return fromHex(color)
        }
    } catch (e: Exception) {
        return Color.Transparent
    }

    return Color.Transparent
}

internal fun hasShadow (styles: BlockStyles): Boolean {
    return styles.boxShadowX != null || styles.boxShadowY != null ||
        styles.boxShadowBlur != null || styles.boxShadowSpread != null
}

internal fun hasBorderRadius (styles: BlockStyles): Boolean {
    return styles.borderTopLeftRadius != null || styles.borderTopRightRadius != null ||
        styles.borderBottomRightRadius != null || styles.borderBottomLeftRadius != null
}

internal fun hasBorder (styles: BlockStyles): Boolean {
    return styles.borderWidth != null
}

fun Modifier.oakShadow(
    color: Color = Color.Black,
    borderRadius: Dp = 0.dp,
    blurRadius: Dp = 0.dp,
    offsetY: Dp = 0.dp,
    offsetX: Dp = 0.dp,
    spread: Dp = 0f.dp,
    modifier: Modifier = Modifier
) = this.then(
    modifier.drawBehind {
        this.drawIntoCanvas {
            val paint = Paint()
            val frameworkPaint = paint.asFrameworkPaint()
            val spreadPixel = spread.toPx()
            val leftPixel = (0f - spreadPixel) + offsetX.toPx()
            val topPixel = (0f - spreadPixel) + offsetY.toPx()
            val rightPixel = (this.size.width + spreadPixel)
            val bottomPixel = (this.size.height + spreadPixel)

            if (blurRadius != 0.dp) {
                frameworkPaint.maskFilter =
                    (BlurMaskFilter(blurRadius.toPx(), BlurMaskFilter.Blur.NORMAL))
            }

            frameworkPaint.color = color.toArgb()
            it.drawRoundRect(
                left = leftPixel,
                top = topPixel,
                right = rightPixel,
                bottom = bottomPixel,
                radiusX = borderRadius.toPx(),
                radiusY = borderRadius.toPx(),
                paint
            )
        }
    }
)
