package tech.poool.commons.db

import android.content.Context
import androidx.datastore.core.DataStore
import androidx.datastore.preferences.core.Preferences
import androidx.datastore.preferences.core.edit
import androidx.datastore.preferences.core.intPreferencesKey
import androidx.datastore.preferences.core.stringPreferencesKey
import androidx.datastore.preferences.preferencesDataStore
import kotlinx.coroutines.flow.first
import java.text.SimpleDateFormat
import java.util.Calendar
import java.util.Locale

val Context.pooolDataStore: DataStore<Preferences> by preferencesDataStore(name = "poool-db")

internal open class BaseDb (
    private val name: String = "generic",
    private val context: Context
) {
    suspend fun getString (key: String): String? {
        return context.pooolDataStore.data
            .first()[stringPreferencesKey("$name:$key")]
    }

    suspend fun getInt (key: String): Int? {
        return context.pooolDataStore.data
            .first()[intPreferencesKey("$name:$key")]
    }

    suspend fun set (key: String, value: String) {
        context.pooolDataStore.edit {
            it[stringPreferencesKey("$name:$key")] = value
        }
    }

    suspend fun set (key: String, value: Int) {
        context.pooolDataStore.edit {
            it[intPreferencesKey("$name:$key")] = value
        }
    }

    suspend fun inc (key: String): Int {
        context.pooolDataStore.edit {
            it[intPreferencesKey("$name:$key")] = (it[intPreferencesKey("$name:$key")] ?: 0) + 1
        }

        return (getInt(key) ?: 0) + 1
    }

    suspend fun getPerDay (key: String, days: Int? = 0): Int {
        var value = 0

        for (i in 0..(days ?: 0)) {
            val day = Calendar.getInstance().apply { add(Calendar.DAY_OF_YEAR, -i) }
            val k = "$key.${formatDate(day)}"

            value += getInt(k) ?: 0
        }

        return value
    }

    fun formatDate (date: Calendar): String {
        return SimpleDateFormat("yyyyMMdd", Locale.US).format(date.time)
    }
}
