package tech.poool.engage.network

import com.google.gson.GsonBuilder
import org.json.JSONArray
import org.json.JSONObject
import tech.poool.commons.network.BaseNetwork
import tech.poool.commons.network.PooolPlatform
import tech.poool.commons.network.utils.RequestBuilder
import tech.poool.commons.oak.data.BlockItem
import tech.poool.commons.oak.data.ColorPaletteDefinition
import tech.poool.commons.oak.network.BlockDeserializer
import tech.poool.commons.oak.network.PaletteDefinitionDeserializer
import tech.poool.engage.BuildConfig
import tech.poool.engage.EngageConfig
import tech.poool.engage.EngageLogger
import tech.poool.engage.network.reponses.AutoCreateResponse
import tech.poool.engage.network.reponses.ConditionDeserializer
import tech.poool.engage.network.reponses.ElementResponse
import tech.poool.engage.network.reponses.EngageElementCondition
import tech.poool.engage.network.reponses.SentResponse
import tech.poool.engage.network.reponses.SuccessResponse

internal class EngageNetwork(
    private val config: EngageConfig,
    logger: EngageLogger,
    private val baseUrl: String = "/api/v3",
): BaseNetwork(
    config = config,
    logger = logger,
    isDebug = BuildConfig.DEBUG
) {
    private val req: RequestBuilder by lazy {
        RequestBuilder(
            config.get("appId"),
            baseUrl,
            BuildConfig.ENGAGE_PUBLIC_KEY
        ).newHeader(PooolPlatform.ACCESS.value.toString())
    }

    private val gson = GsonBuilder()
        .registerTypeAdapter(BlockItem::class.java, BlockDeserializer())
        .registerTypeAdapter(EngageElementCondition::class.java, ConditionDeserializer())
        .registerTypeAdapter(ColorPaletteDefinition::class.java, PaletteDefinitionDeserializer())
        .create()

    suspend fun getEngageHealth(): String? {
        val request = req.newBuilder()
            .addEndpoint("/engage/health")
            .build()

        val res = this.request(request)

        return res?.body?.string()
    }

    suspend fun getElement(slug: String, signature: Boolean = true): ElementResponse? {
        val jsonAdapter = gson.getAdapter(ElementResponse::class.java)
        val request = req.newBuilder()
            .addEndpoint("/engage/element/$slug")
            .addMethod("GET")
            .useSignature(signature)
            .build()

        val res = this.request(request)

        return res?.body?.string()?.let {
            jsonAdapter.fromJson(it)
        }
    }

    suspend fun autoCreate(filters: List<String>? = null, signature: Boolean = true): AutoCreateResponse? {
        val jsonAdapter = gson.getAdapter(AutoCreateResponse::class.java)
        val request = req.newBuilder()
            .addEndpoint("/engage/elements")
            .addMethod("POST")
            .addBody(JSONObject("""{
                "filters": ${JSONArray(filters ?: listOf<String>())},
                "url": "test",
                "referrer": "test"
            }""".trimMargin()))
            .useSignature(signature)
            .build()

        val res = this.request(request)

        return res?.body?.string()?.let {
            jsonAdapter.fromJson(it)
        }
    }

    suspend fun sendEvent(data: Map<String, *>, signature: Boolean = true): SuccessResponse? {
        val jsonAdapter = gson.getAdapter(SuccessResponse::class.java)
        val request = req.newBuilder()
            .addEndpoint("/engage/event")
            .addMethod("POST")
            .useSignature(signature)
            .addBody(JSONObject(data))
            .build()

        val res = this.request(request)

        return res?.body?.string()?.let {
            jsonAdapter.fromJson(it)
        }
    }

    suspend fun submitForm(data: Map<String, Any>, signature: Boolean = true): SentResponse? {
        val jsonAdapter = gson.getAdapter(SentResponse::class.java)
        val request = req.newBuilder()
            .addEndpoint("/engage/form/submit")
            .addMethod("POST")
            .useSignature(signature)
            .addBody(JSONObject(data))
            .build()

        val res = this.request(request)

        return res?.body?.string()?.let {
            jsonAdapter.fromJson(it)
        }
    }
}
