package tech.poool.engage.compose

import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.runtime.mutableStateMapOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.snapshots.SnapshotStateMap
import androidx.compose.ui.platform.LocalContext
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import tech.poool.engage.network.reponses.EngageElement
import tech.poool.engage.core.shared.LocalEngage
import tech.poool.engage.Engage

@Composable
fun EngageProvider(
    appId: String,
    config: Map<String, Any?> = emptyMap(),
    variables: Map<String, Any?> = emptyMap(),
    texts: Map<String, String?> = emptyMap(),
    content: @Composable () -> Unit,
) {
    val context = LocalContext.current
    val (engage) = remember { mutableStateOf(
        Engage(
            appId,
            context = context,
            mode = Engage.MODE_COMPOSE,
        )
            .config(config)
            .variables(variables)
            .texts(texts)
    ) }
    val targets = remember { mutableStateMapOf<String, List<EngageElement>>() }
    val coroutineScope = rememberCoroutineScope()

    fun sendEvent (data: Map<String, *>) {
        coroutineScope.launch {
            engage.manager.engageViewModel.sendEvent(data).await()
        }
    }

    CompositionLocalProvider(
        LocalEngage provides EngageProviderData(
            engage = engage,
            targets = targets,
            scope = coroutineScope,
            config = config,
            variables = variables,
            texts = texts,
            sendEvent = ::sendEvent
        ),
        content = content
    )
}

internal data class EngageProviderData (
    val engage: Engage? = null,
    val targets: SnapshotStateMap<String, List<EngageElement>> = SnapshotStateMap(),
    val scope: CoroutineScope? = null,
    val config: Map<String, Any?>? = null,
    val variables: Map<String, Any?>? = null,
    val texts: Map<String, String?>? = null,
    val sendEvent: (data: Map<String, *>) -> Unit = {}
)
