package tech.poool.engage.compose

import androidx.activity.ComponentActivity
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Close
import androidx.compose.material3.BottomSheetDefaults
import androidx.compose.material3.Card
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.ModalBottomSheet
import androidx.compose.material3.rememberModalBottomSheetState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.layout.boundsInParent
import androidx.compose.ui.layout.boundsInWindow
import androidx.compose.ui.layout.onGloballyPositioned
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalSoftwareKeyboardController
import androidx.compose.ui.unit.dp
import androidx.compose.ui.window.Dialog
import kotlinx.coroutines.launch
import tech.poool.commons.compose.CommonsProvider
import tech.poool.commons.compose.LocalForm
import tech.poool.commons.oak.components.Block
import tech.poool.engage.ElementData
import tech.poool.engage.EngageEvents
import tech.poool.engage.FieldError
import tech.poool.engage.core.fieldBlocks
import tech.poool.engage.network.reponses.EngageElement
import tech.poool.engage.core.shared.LocalElement
import tech.poool.engage.core.shared.LocalEngage
import tech.poool.commons.oak.components.FormProvider
import tech.poool.commons.oak.components.FormProviderData
import tech.poool.commons.oak.data.BlockFieldItem
import tech.poool.commons.oak.getBlocksByType
import tech.poool.engage.network.reponses.EngageElementBottomSheetCondition
import tech.poool.engage.network.reponses.EngageElementModalCondition

@OptIn(ExperimentalMaterial3Api::class)
@Composable
internal fun Element (
    modifier: Modifier = Modifier,
    element: EngageElement,
) {
    val componentActivity = LocalContext.current as? ComponentActivity
    val engageProvider = LocalEngage.current
    val engageViewModel = engageProvider.engage?.manager?.engageViewModel
    val engageDataConfig = engageViewModel?.dataConfig
    val keyboard = LocalSoftwareKeyboardController.current
    val currentScope = engageProvider.scope
    var isReady by remember { mutableStateOf(false) }
    var isSeen by remember { mutableStateOf(false) }
    var isClosed by remember { mutableStateOf(false) }

    val bottomSheetCondition = element.conditions
        ?.find { it.type == "bottom-sheet" } as EngageElementBottomSheetCondition?
    val modalCondition = element.conditions
        ?.find { it.type == "modal" } as EngageElementModalCondition?

    val bottomSheetState = if (bottomSheetCondition?.value?.closable == true) {
        rememberModalBottomSheetState()
    } else {
        rememberModalBottomSheetState(
            skipPartiallyExpanded = true,
            confirmValueChange = { false }
        )
    }

    fun onSubmit (form: FormProviderData) {
        form.setLoading(true)

        val formFields = getBlocksByType(fieldBlocks, element.content) as MutableList<BlockFieldItem>
        val validationErrors = form.getValidationErrors(formFields)

        if (validationErrors.isNotEmpty()) {
            engageDataConfig?.logger?.w("Form validation errors found:", validationErrors.toString())
            form.setLoading(false)
            form.setErrors(validationErrors)
            return
        }

        form.setErrors(null)
        keyboard?.hide()
        currentScope?.let { it.launch {
            val results = engageDataConfig?.events?.emit(EngageEvents.FORM_SUBMIT, mapOf(
                "fields" to formFields,
                "values" to form.values,
                "element" to ElementData(
                    name = element.name,
                    slug = element.slug,
                )
            )) as List<List<FieldError?>?>? ?: mutableListOf<List<FieldError?>>()

            val errors = results.filterNotNull().flatten().filterNotNull()

            if (errors.isNotEmpty()) {
                engageDataConfig?.logger?.w("Form errors found:", errors.toString())
                form.setLoading(false)
                form.setErrors(errors)

                return@launch
            }

            val result = engageViewModel?.submitForm(mapOf(
                "elementSlug" to element.slug,
                "fields" to formFields.map { field -> mapOf(
                    "fieldKey" to field.fieldKey,
                    "fieldType" to field.fieldType,
                    "fieldValue" to form.values[field.fieldKey ?: "unknown"],
                ) },
                "pageUrl" to "",
            ))?.await()

            if (result?.errors != null) {
                form.setErrors(result.errors.fields)
            } else if (result?.sent == true) {
                bottomSheetState.hide()
                form.setSent(true)
                isClosed = true
            }

            engageViewModel?.sendEvent(mapOf(
                "type" to "form-submit",
                "url" to "",
                "element" to element.id,
            )).run {}

            form.setLoading(false)
        } } ?: run {
            engageDataConfig?.logger?.w("No coroutine scope found to submit the form")
            form.setLoading(false)
        }
    }

    fun onReady () {
        isReady  = true

        currentScope?.launch {
            engageDataConfig?.events?.emit(EngageEvents.READY, mapOf(
                "element" to mapOf(
                    "name" to element.name,
                    "slug" to element.slug,
                )
            ))

            engageViewModel?.sendEvent(mapOf(
                "type" to "impression",
                "url" to "",
                "element" to element.id,
            )).run {  }
        }
    }

    fun onSeen () {
        isSeen = true

        currentScope?.launch {
            engageDataConfig?.events?.emit(EngageEvents.SEEN, mapOf(
                "element" to ElementData(
                    name = element.name,
                    slug = element.slug,
                )
            ))

            engageViewModel?.sendEvent(mapOf(
                "type" to "view",
                "url" to "",
                "element" to element.id,
            )).run{ }
        }
    }

    fun onDestroy (buttonId: String?) {
        if (isClosed) {
            return
        }

        isClosed = true

        currentScope?.launch {
            engageDataConfig?.events?.emit(EngageEvents.DESTROY, mapOf(
                "buttonId" to buttonId,
                "element" to ElementData(
                    name = element.name,
                    slug = element.slug,
                )
            ))
        }

        if (
            (bottomSheetCondition != null && !bottomSheetCondition.value.closable) ||
            (modalCondition != null && !modalCondition.value.closable)
        ){
            componentActivity?.onBackPressedDispatcher?.onBackPressed()
        }
    }

    fun onClick (data: Map<String, Any?>?) {
        currentScope?.launch {
            engageViewModel?.sendEvent((data ?: emptyMap()).plus(mapOf(
                "type" to "click",
                "element" to element.id,
            ))).run{ }
        }
    }

    fun onTriggerEvent (eventType: String, eventData: Map<String, Any?>?) {
        engageDataConfig?.events
            ?.emitSync(EngageEvents.valueOf(eventType), eventData ?: emptyMap())
    }

    LaunchedEffect (Unit) {
        if (!isReady) {
            onReady()
        }
    }

    if (isClosed) {
        return
    }

    val content = @Composable {
        CompositionLocalProvider(
            LocalElement provides ElementProviderData(
                element = element
            ),
        ) {
            CommonsProvider (
                config = engageDataConfig?.config,
                texts = engageDataConfig?.texts,
                variables = engageDataConfig?.variables,
                logger = engageDataConfig?.logger,
            ) {
                FormProvider(onSubmit = ::onSubmit) {
                    Column(
                        modifier = modifier
                            .fillMaxWidth()
                            .wrapContentHeight()
                            .onGloballyPositioned { coordinates ->
                                val rect = coordinates.boundsInWindow()
                                val rectInParent = coordinates.boundsInParent()

                                val percentageVisible = (rect.height / rectInParent.height) * 100

                                if (percentageVisible >= 30 && !isSeen) {
                                    onSeen()
                                }
                            }
                    ) {
                        element.content.forEach {
                            EngageBlock(
                                block = it,
                                element = element,
                                onDestroy = ::onDestroy,
                                onClick = ::onClick,
                                onTriggerEvent = ::onTriggerEvent,
                            )
                        }
                    }
                }
            }
        }
    }

    if (bottomSheetCondition != null) {
        ModalBottomSheet(
            dragHandle = {
                if (!bottomSheetCondition.value.closable) return@ModalBottomSheet
                else BottomSheetDefaults.DragHandle()
            },
            onDismissRequest = {
                if (!bottomSheetCondition.value.closable) {
                    componentActivity?.onBackPressedDispatcher?.onBackPressed()
                }

                isClosed = true
            },
            sheetState = bottomSheetState
        ) {
            content()
        }
    } else if (modalCondition != null) {
        Dialog(
            onDismissRequest = {
                if (!modalCondition.value.closable) {
                    componentActivity?.onBackPressedDispatcher?.onBackPressed()
                    return@Dialog
                }

                isClosed = true
            }
        ) {
            Card (
                modifier = Modifier
                    .fillMaxWidth()
                    .wrapContentHeight()
            ) {
                Box {
                    if (modalCondition.value.closable) {
                        Row (
                            verticalAlignment = Alignment.CenterVertically,
                            horizontalArrangement = Arrangement.End,
                            modifier = Modifier
                                .fillMaxWidth()
                                .padding(vertical = 4.dp),
                        ) {
                            Icon(
                                imageVector = Icons.Default.Close,
                                contentDescription = "Close",
                                modifier = Modifier
                                    .padding(end = 8.dp)
                                    .clickable { isClosed = true }
                            )
                        }
                    }

                    Box (modifier = Modifier
                        .fillMaxWidth()
                        .wrapContentHeight()
                        .padding(16.dp)
                    ) {
                        content()
                    }
                }
            }
        }
    } else {
        content()
    }
}

internal data class ElementProviderData (
    val element: EngageElement? = null,
)
