package tech.poool.commons.oak.components

import android.util.Patterns
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.runtime.mutableStateMapOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.snapshots.SnapshotStateMap
import tech.poool.commons.compose.LocalForm
import tech.poool.commons.oak.data.*

@Composable
internal fun FormProvider (
    onSubmit: (values: FormProviderData) -> Unit,
    content: @Composable () -> Unit,
) {
    val values = remember { mutableStateMapOf<String, Any>() }
    val valid = remember { mutableStateMapOf<String, Boolean>() }

    val (errors, setErrors) = remember { mutableStateOf<List<FieldError>?>(null) }
    val (isLoading, setLoading) = remember { mutableStateOf(false) }
    val (isSent, setSent) = remember { mutableStateOf(false) }

    fun getValidationErrors (fields: List<BlockItem>): List<FieldError> {
        val result = mutableListOf<FieldError>()

        fields.filter { it is BlockFieldItem && it.fieldRequired ?: false }.forEach {
            if (it is BlockFieldItem) {
                if (valid[it.fieldKey] != true) {
                    result.add(BaseFieldError(
                        fieldKey = it.fieldKey ?: "unknown",
                        message = when (it.fieldType) {
                            "email" -> "form_email_error"
                            else -> "form_empty_error"
                        }
                    ))
                }
            }
        }

        return result
    }

    fun onTextChange (field: BlockFieldItem, value: String) {
        if (errors != null) setErrors(null)

        values[field.fieldKey ?: "unknown"] = value
        valid[field.fieldKey ?: "unknown"] = if (field.fieldRequired != false) value.isNotEmpty() else true

        when (field.fieldType) {
            "email" -> {
                if (!Patterns.EMAIL_ADDRESS.matcher(value).matches()) {
                    valid[field.fieldKey ?: "unknown"] = false
                }
            }
        }
    }

    fun onToggle (field: BlockFieldItem, value: Boolean) {
        if (errors != null) setErrors(null)

        values[field.fieldKey ?: "unknown"] = value
        valid[field.fieldKey ?: "unknown"] = if (field.fieldRequired != false) value else true
    }

    fun onInternalSubmit () {
        onSubmit(FormProviderData(
            values = values,
            valid = valid,
            errors = errors,
            isLoading = isLoading,
            isSent = isSent,
            onTextChange = ::onTextChange,
            onToggle = ::onToggle,
            onSubmit = ::onInternalSubmit,
            setLoading = setLoading,
            setSent = setSent,
            setErrors = setErrors,
            getValidationErrors = ::getValidationErrors,
        ))
    }

    CompositionLocalProvider(
        value = LocalForm provides FormProviderData(
            values = values,
            valid = valid,
            errors = errors,
            isLoading = isLoading,
            isSent = isSent,
            onTextChange = ::onTextChange,
            onToggle = ::onToggle,
            onSubmit = ::onInternalSubmit,
            setLoading = setLoading,
            setSent = setSent,
            setErrors = setErrors,
            getValidationErrors = ::getValidationErrors,
        ),
        content = content
    )

}

internal data class FormProviderData(
    var values: SnapshotStateMap<String, Any> = SnapshotStateMap(),
    var valid: SnapshotStateMap<String, Boolean> = SnapshotStateMap(),
    var errors: List<FieldError>? = null,
    var isLoading: Boolean = false,
    var isSent: Boolean = false,
    var onTextChange: (field: BlockFieldItem, value: String) -> Unit = { _, _ -> },
    var onToggle: (field: BlockFieldItem, value: Boolean) -> Unit = { _, _ -> },
    var onSubmit: () -> Unit = { },
    var setLoading: (Boolean) -> Unit = { },
    var setSent: (Boolean) -> Unit = { },
    var setErrors: (List<FieldError>?) -> Unit = { },
    var getValidationErrors: (fields: List<BlockItem>) -> List<FieldError> = { emptyList() },
)
