package tech.poool.commons.oak

import android.content.Context
import android.util.TypedValue
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.TileMode
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import tech.poool.commons.oak.data.BlockClickableItem
import tech.poool.commons.oak.data.BlockColItem
import tech.poool.commons.oak.data.BlockFoldableItem
import tech.poool.commons.oak.data.BlockItem
import tech.poool.commons.oak.data.BlockRowItem
import tech.poool.commons.oak.data.BlockStyles

internal fun fromHex(hex: String): Color {
    var result = hex

    if (hex.startsWith("#") && hex.length == 4) {
        val red = hex.substring(1, 2)
        val green = hex.substring(2, 3)
        val blue = hex.substring(3, 4)
        result = "#$red$red$green$green$blue$blue"
    } else if (hex.startsWith("rgb(")) {
        val rgb = hex.substring(4, hex.length - 1).split(",")
        val red = rgb[0].trim().toInt()
        val green = rgb[1].trim().toInt()
        val blue = rgb[2].trim().toInt()
        result = String.format("#%02x%02x%02x", red, green, blue)
    } else if (hex.startsWith("rgba(")) {
        val rgba = hex.substring(5, hex.length - 1).split(",")
        val red = rgba[0].trim().toInt()
        val green = rgba[1].trim().toInt()
        val blue = rgba[2].trim().toInt()
        val alpha = rgba[3].trim().toFloat()
        result = String.format("#%02x%02x%02x%02x", red, green, blue, (alpha * 255).toInt())
    }

    return try {
        Color(android.graphics.Color.parseColor(result))
    } catch (e: Exception) {
        Color.Transparent
    }
}

/**
 * Oak sizes can be "2", "2px", "2%", and all the other css sizes (like em, rem, ...) but fuck that
 */
internal fun oakSizeToDp (
    size: String?
): Dp {
    return size?.filter { it.isDigit() }?.toIntOrNull()?.dp ?: 0.dp
}

internal fun oakToTileModeX(
    styles: BlockStyles? = null
): TileMode {
    return when (styles?.backgroundRepeat) {
        "repeat-x" -> TileMode.Repeated
        "repeat" -> TileMode.Repeated
        else -> TileMode.Decal
    }
}

internal fun oakToTileModeY(
    styles: BlockStyles? = null
): TileMode {
    return when (styles?.backgroundRepeat) {
        "repeat-y" -> TileMode.Repeated
        "repeat" -> TileMode.Repeated
        else -> TileMode.Decal
    }
}

internal fun oakToBackgroundSize(
    imageSize: Size,
    parentSize: Size,
    backgroundSize: String,
): Size {
    val mode = getAspectRatioMode(imageSize)
    val parentMode = getAspectRatioMode(parentSize)

    val aspectRatio = imageSize.width / imageSize.height
    val aspectRatioReverse = imageSize.height / imageSize.width

    val aspectRatioedWidth = Size(parentSize.height * aspectRatio, parentSize.height)
    val aspectRatioedHeight = Size(parentSize.width, parentSize.width * aspectRatioReverse)

    // We're too tired to comment, deal with it
    // TODO: Add comments
    return when (backgroundSize) {
        "cover" -> when(parentMode) {
            "square" -> when (mode) {
                "square" -> parentSize
                "landscape" -> aspectRatioedWidth
                "portrait" -> aspectRatioedHeight
                else -> imageSize
            }
            "landscape" -> when (mode) {
                "square", "portrait" -> aspectRatioedHeight
                "landscape" ->
                    if (aspectRatioedWidth.height == parentSize.height && aspectRatioedWidth.width < parentSize.width)
                        aspectRatioedHeight
                    else
                        aspectRatioedWidth
                else -> imageSize
            }
            "portrait" -> when (mode) {
                "square", "landscape" -> aspectRatioedWidth
                "portrait" ->
                    if (aspectRatioedHeight.width == parentSize.width && aspectRatioedHeight.height < parentSize.height)
                        aspectRatioedWidth
                    else
                        aspectRatioedHeight
                else -> imageSize
            }
            else -> imageSize
        }
        "contain" -> when(parentMode) {
            "square" -> when (mode) {
                "square" -> parentSize
                "landscape" -> aspectRatioedHeight
                "portrait" -> aspectRatioedWidth
                else -> imageSize
            }
            "landscape" -> when (mode) {
                "square", "portrait" -> aspectRatioedWidth
                "landscape" ->
                    if (aspectRatioedHeight.width == parentSize.width && aspectRatioedHeight.height < parentSize.height)
                        aspectRatioedHeight
                    else
                        aspectRatioedWidth
                else -> imageSize
            }
            "portrait" -> when (mode) {
                "square", "landscape" -> aspectRatioedHeight
                "portrait" ->
                    if (aspectRatioedWidth.height == parentSize.height && aspectRatioedWidth.width < parentSize.width)
                        aspectRatioedWidth
                    else
                        aspectRatioedHeight
                else -> imageSize
            }
            else -> imageSize
        }
        else -> imageSize
    }
}

internal fun getAspectRatioMode(size: Size): String {
    return if (size.width == size.height) "square"
        else if (size.width > size.height) "landscape"
        else "portrait"
}

internal fun oakToBackgroundPosition(
    imageSize: Size,
    parentSize: Size,
    position: String,
): Offset {
    return when (position) {
        "center" -> Offset(
            (parentSize.width - imageSize.width) / 2,
            (parentSize.height - imageSize.height) / 2
        )
        "left top" -> Offset(0f, 0f)
        "right top" -> Offset(parentSize.width - imageSize.width, 0f)
        "center top" -> Offset((parentSize.width - imageSize.width) / 2, 0f)
        "left center" -> Offset(0f, (parentSize.height - imageSize.height) / 2)
        "right center" -> Offset(parentSize.width - imageSize.width, (parentSize.height - imageSize.height) / 2)
        "center bottom" -> Offset((parentSize.width - imageSize.width) / 2, parentSize.height - imageSize.height)
        "left bottom" -> Offset(0f, parentSize.height - imageSize.height)
        "right bottom" -> Offset(parentSize.width - imageSize.width, parentSize.height - imageSize.height)
        else -> Offset.Zero
    }
}

/**
 * Android TextView needs px for things like line height, where compose uses sp
 */
internal fun spToPx(sp: Float, context: Context): Int {
    return TypedValue.applyDimension(
        TypedValue.COMPLEX_UNIT_SP,
        sp,
        context.resources.displayMetrics
    ).toInt()
}

internal fun getBlocksByType (
    types: List<String>,
    content: List<BlockItem>
): MutableList<BlockItem> {
    return content.fold(mutableListOf()) { acc, block ->
        if (block is BlockRowItem) {
            acc.addAll(getBlocksByType(types, block.cols))
        } else if (block is BlockColItem) {
            acc.addAll(getBlocksByType(types, block.content))
        } else if (block is BlockClickableItem) {
            acc.addAll(getBlocksByType(types, block.content))
        } else if (block is BlockFoldableItem) {
            acc.addAll(getBlocksByType(types, block.content))
            acc.addAll(getBlocksByType(types, block.seeLess))
            acc.addAll(getBlocksByType(types, block.seeMore))
        } else if (types.contains(block.type)) {
            acc.add(block)
        }

        return@fold acc
    }
}

internal fun filterBlocks (
    content: List<BlockItem>,
    discriminator: (BlockItem) -> Boolean
): MutableList<BlockItem> {
    return content.fold(mutableListOf()) { acc, block ->
        if (block is BlockRowItem) {
            acc.addAll(filterBlocks(block.cols, discriminator))
        } else if (block is BlockColItem) {
            acc.addAll(filterBlocks(block.content, discriminator))
        } else if (block is BlockClickableItem) {
            acc.addAll(filterBlocks(block.content, discriminator))
        } else if (block is BlockFoldableItem) {
            acc.addAll(filterBlocks(block.content, discriminator))
            acc.addAll(filterBlocks(block.seeLess, discriminator))
            acc.addAll(filterBlocks(block.seeMore, discriminator))
        } else if (discriminator(block)) {
            acc.add(block)
        }

        return@fold acc
    }
}
