package tech.poool.commons.network

import java.util.concurrent.TimeUnit
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import okhttp3.Interceptor
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.Response
import okhttp3.logging.HttpLoggingInterceptor
import tech.poool.commons.config.BaseConfig
import tech.poool.commons.logger.BaseLogger
import tech.poool.commons.network.utils.CookieManager

internal open class BaseNetwork(
    config: BaseConfig,
    private val logger: BaseLogger,
    isDebug: Boolean = false,
) {
    private val okHttp: OkHttpClient
    private var interceptors: MutableList<Interceptor> = mutableListOf()
    private val isCookieEnabled = config.get("cookiesEnabled") ?: false

    init {
        val cookieManager = CookieManager(isCookieEnabled, logger)
        val builder = OkHttpClient.Builder()
            .connectTimeout(10, TimeUnit.SECONDS)
            .readTimeout(10, TimeUnit.SECONDS)
            .retryOnConnectionFailure(true)

        isCookieEnabled.let {
            if (it) builder.cookieJar(cookieManager)
        }

        if (isDebug) {
            val logger = HttpLoggingInterceptor()
                .setLevel(HttpLoggingInterceptor.Level.BASIC)

            interceptors.add(logger)
        }

        interceptors.forEach { builder.addInterceptor(it) }

        okHttp = builder.build()
    }

    suspend fun request(request: Request): Response? {
        return withContext(Dispatchers.IO) {
            var response: Response? = null

            try {
                response = okHttp.newCall(request).execute()

                if (!response.isSuccessful) {
                    val responseBodyCopy = response.peekBody(Long.MAX_VALUE)
                    val responseBody = responseBodyCopy.string()

                    logger.w(responseBody, SCOPE)

                    response = null
                }
            } catch (e: Exception) {
                logger.e(e.message ?: e.toString(), SCOPE)
            }

            return@withContext response
        }
    }

    companion object {
        const val SCOPE = "NETWORK"
    }
}
