package tech.poool.commons.events

import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import tech.poool.commons.logger.BaseLogger

internal interface BaseEventTypes {
    val name: String
}

internal open class BaseEvents<T: BaseEventTypes> (
    private val logger: BaseLogger? = null
) {
    private val events: MutableMap<
        T,
        MutableList<Pair<Boolean, suspend (Map<String, Any?>) -> Any?>>
    > = mutableMapOf()

    private val eventsSync: MutableMap<
        T,
        MutableList<Pair<Boolean, (Map<String, Any?>) -> Any?>>
    > = mutableMapOf()

    fun listen(
        event: T,
        once: Boolean = false,
        callbackSync: ((Map<String, Any?>) -> Any?)? = null,
        callback: (suspend (Map<String, Any?>) -> Any?)? = null,
    ) {
        logger?.i("Listening to <${event.name}> event")

        if (callback == null && callbackSync == null) {
            return
        }

        if (!eventsSync.containsKey(event) && !events.containsKey(event)) {
            callbackSync?.let { eventsSync[event] = mutableListOf(once to it) }
            callback?.let { events[event] = mutableListOf(once to it) }

            return
        }

        callbackSync?.let { eventsSync.getOrPut(event, ::mutableListOf).add(once to it) }
        callback?.let { events.getOrPut(event, ::mutableListOf).add(once to it) }
    }

    fun emitSync(
        event: T,
        data: Map<String, Any?> = mapOf(),
    ): MutableList<Any?> {
        logger?.i("Emitting <${event.name}> event")

        val results = mutableListOf<Any?>()

        if (!eventsSync.containsKey(event)) return results

        eventsSync[event]?.forEachIndexed { index, pair ->
            val (once, callback) = pair

            try {
                results.add(callback(data))
            } catch (e: Exception) {
                e.printStackTrace()
            }

            logger?.i("Fired <${event.name}> event listener" +
                if (once) " (once, then muted)" else "")

            if (once) {
                eventsSync[event] = eventsSync[event]
                    ?.filterIndexed { i, _ -> i != index }
                    ?.toMutableList() ?: mutableListOf()
            }
        }

        return results
    }

    suspend fun emit(
        event: T,
        data: Map<String, Any?> = mapOf()
    ): MutableList<Any?> {
        return withContext(Dispatchers.IO) {
            logger?.i("Emitting <${event.name}> event")

            val results = mutableListOf<Any?>()

            if (!events.containsKey(event)) return@withContext results

            events[event]?.forEachIndexed { index, pair ->
                val (once, callback) = pair

                try {
                    results.add(callback(data))
                } catch (e: Exception) {
                    e.printStackTrace()
                }

                logger?.i("Fired <${event.name}> event listener" +
                    if (once) " (once, then muted)" else "")

                if (once) {
                    events[event] = events[event]
                        ?.filterIndexed { i, _ -> i != index }
                        ?.toMutableList() ?: mutableListOf()
                }
            }

            return@withContext results
        }
    }

    fun mute(event: T) {
        if (!events.containsKey(event) && !eventsSync.containsKey(event)) {
            logger?.i("Could not find corresponding event handler to be muted for " +
                "event <${event.name}>, skipping")

            return
        }

        events[event]?.clear()
        eventsSync[event]?.clear()

        events[event]?.let { if (events[event]?.size!! < 1) events.remove(event) }
        eventsSync[event]?.let { if (eventsSync[event]?.size!! < 1) eventsSync.remove(event) }

        logger?.i("Muting <${event.name}> event")
    }

    fun getAll(): Map<
        T,
        MutableList<Pair<Boolean, suspend (Map<String, Any?>) -> Any?>>
    > {
        return events
    }

    fun reset() {
        events.clear()
        eventsSync.clear()
    }

    companion object {
        const val CLICK = "click"
    }
}
