package tech.ostack.kform.validations

import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import tech.ostack.kform.Validation
import tech.ostack.kform.ValidationContext
import tech.ostack.kform.ValidationIssue
import tech.ostack.kform.ValidationIssueSeverity
import tech.ostack.kform.datatypes.Table
import tech.ostack.kform.datatypes.TableRow
import tech.ostack.kform.validations.UniqueItems.Companion.DEFAULT_CODE
import tech.ostack.kform.validations.UniqueItemsBy.Companion.DEFAULT_CODE

/**
 * Validation that ensures that a value does not contain repeated items, where the uniqueness of an
 * item is determined by its key as returned by [selector]. Values of type [Collection], [Array]
 * (including variants), and [Table] are supported.
 *
 * Depending on [emitAllRepetitions], an issue with the provided [code] (defaults to [DEFAULT_CODE])
 * is emitted for each pair of repeated items (when `true`), or only for the first found repetition
 * (when `false`). [emitAllRepetitions] defaults to `true`.
 *
 * Each emitted issue has a `firstIndex` and `secondIndex` data properties with the two indices of
 * the conflicting items. When running this validation on a table, issues will also contain
 * `firstRowId` and `secondRowId` data properties with the ids of the conflicting table rows.
 *
 * [Pair]s, [Triple]s, and [List]s work well to represent composite keys. E.g. say that you have a
 * value of items that should be unique in respect to their fields `A` and `B`; you can use
 * `UniqueBy` as such:
 * ```kotlin
 * UniqueBy { Pair(it.A, it.B) }
 * ```
 *
 * When the result of calling [selector] on an item is `null`, that item is always considered
 * unique. I.e. if `selector(A) == null` and `selector(B) == null`, then `A` is considered different
 * to `B`.
 *
 * If you wish to treat multiple `null` values as equal to one another, consider wrapping said
 * values in an object (e.g. a list with a single item).
 *
 * This validation is **not** stateful and depends on all descendants of the value.
 *
 * @property emitAllRepetitions Whether to emit an issue per each repeated item or only for the
 *   first repetition.
 * @property code Issue code to use when two items of the value are repeated.
 * @property severity Severity of the issue emitted when two items of the value are repeated.
 * @property selector Selector function used to specify the key of an item of the value.
 */
public open class UniqueItemsBy<T, TKey>
@JvmOverloads
constructor(
    public val emitAllRepetitions: Boolean = true,
    public val code: String = DEFAULT_CODE,
    public val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
    @param:BuilderInference public val selector: (item: T) -> TKey?,
) : Validation<Any>() {
    override fun toString(): String = "UniqueItemsBy"

    override val dependsOnDescendants: Boolean = true

    @Suppress("UNCHECKED_CAST")
    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        val indices = HashMap<TKey, Int>(size(value))
        for ((i, el) in iterableWithIndex(value)) {
            val item = (if (el is TableRow<*>) el.value else el) as T
            val key = selector(item)
            if (key != null) {
                val conflictIndex = indices[key]
                if (conflictIndex != null) {
                    emit(
                        ValidationIssue(
                            code,
                            severity,
                            buildMap {
                                put("firstIndex", "$conflictIndex")
                                put("secondIndex", "$i")
                                if (el is TableRow<*>) {
                                    put("firstRowId", "${(value as Table<*>).idAt(conflictIndex)}")
                                    put("secondRowId", "${el.id}")
                                }
                            },
                        )
                    )
                    if (!emitAllRepetitions) {
                        break
                    }
                } else {
                    indices[key] = i
                }
            }
        }
    }

    public companion object {
        /** Default issue code representing that two items of the value are repeated. */
        public const val DEFAULT_CODE: String = "itemsRepeated"
    }
}

/**
 * Validation that ensures that a value does not contain repeated items. Values of type
 * [Collection], [Array] (including variants), and [Table] are supported.
 *
 * Depending on [emitAllRepetitions], an issue with the provided [code] (defaults to [DEFAULT_CODE])
 * is emitted for each pair of repeated items (when `true`), or only for the first found repetition
 * (when `false`). [emitAllRepetitions] defaults to `true`.
 *
 * Each emitted issue has a `firstIndex` and `secondIndex` data properties with the two indices of
 * the conflicting items. When running this validation on a table, issues will also contain
 * `firstRowId` and `secondRowId` data properties with the ids of the conflicting table rows.
 *
 * Depending on [treatNullAsUnique], `null` values can be considered unique. I.e. when `true` (the
 * default), the value `listOf(null, null)` is considered to **not** contain repeated items.
 *
 * This validation is **not** stateful and depends on all descendants of the value.
 *
 * @param emitAllRepetitions Whether to emit an issue per each repeated item or only for the first
 *   repetition.
 * @param code Issue code to use when two items of the value are repeated.
 * @param severity Severity of the issue emitted when two items of the value are repeated.
 * @property treatNullAsUnique Whether to treat `null` values as being unique.
 */
public open class UniqueItems
@JvmOverloads
constructor(
    emitAllRepetitions: Boolean = true,
    public val treatNullAsUnique: Boolean = true,
    code: String = DEFAULT_CODE,
    severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
) :
    UniqueItemsBy<Any?, Any?>(
        emitAllRepetitions,
        code,
        severity,
        { if (treatNullAsUnique) it else listOf(it) },
    ) {
    override fun toString(): String = "UniqueItems"

    public companion object {
        /** Default issue code representing that two items of the value are repeated. */
        public const val DEFAULT_CODE: String = "itemsRepeated"
    }
}

// private typealias StatefulUniqueState<T> = Pair<MutableMap<T, MutableList<Int>>, MutableList<T>>
//
// public open class StatefulUniqueBy<T, TKey>(
//    public val code: String = DEFAULT_CODE,
//    public val emitAllRepetitions: Boolean = true,
//    @BuilderInference public val selector: (T) -> TKey?,
// ) : StatefulValidation<Collection<T>, StatefulUniqueState<TKey>>() {
//    override suspend fun ValidationContext.initState(value: Collection<T>):
// StatefulUniqueState<TKey>
// {
//        val keyIndices = HashMap<TKey, MutableList<Int>>(value.size)
//        val repeatedKeys = mutableListOf<TKey>()
//        for ((i, el) in value.withIndex()) {
//            val key = selector(el)
//            if (key != null) {
//                val indices = keyIndices.getOrPut(key) { mutableListOf() }
//                indices += i
//                if (indices.size == 2) {
//                    repeatedKeys += key
//                }
//            }
//        }
//        return Pair(keyIndices, repeatedKeys)
//    }
//
//    override fun ValidationContext.validateFromState(
//        value: Collection<T>,
//        state: StatefulUniqueState<TKey>
//    ): Flow<Issue> = flow {
//        val (keyIndices, repeatedKeys) = state
//        for (key in repeatedKeys) {
//            val indices = keyIndices[key] ?: error("Invalid 'StatefulUniqueBy' validation state.")
//            for (i in 0..<indices.size - 1) {
//                emit(
//                    Error(
//                        code,
//                        data =
//                            mapOf(
//                                "firstIndex" to "${indices[i]}",
//                                "secondIndex" to "${indices[i + 1]}"
//                            )
//                    )
//                )
//                if (!emitAllRepetitions) {
//                    return@flow
//                }
//            }
//        }
//    }
//
//    public companion object {
//        /** Default issue code representing that two items of the value are repeated. */
//        public const val DEFAULT_CODE: String = "itemsRepeated"
//    }
// }

/** Size of a [Collection], [Array] (including variants), or [Table]. */
private fun size(value: Any): Int =
    when (value) {
        is Collection<*> -> value.size
        is Array<*> -> value.size
        is Table<*> -> value.size
        // Array variants
        is BooleanArray -> value.size
        is ByteArray -> value.size
        is CharArray -> value.size
        is DoubleArray -> value.size
        is FloatArray -> value.size
        is IntArray -> value.size
        is LongArray -> value.size
        is ShortArray -> value.size
        else ->
            error(
                "Unsupported value type: supported types are `Collection`, `Array` (and " +
                    "variants), and `Table`."
            )
    }

/** Iterable with index over a [Collection], [Array] (including variants), or [Table]. */
private fun iterableWithIndex(value: Any): Iterable<IndexedValue<Any?>> =
    when (value) {
        is Collection<*> -> value.withIndex()
        is Array<*> -> value.withIndex()
        is Table<*> -> value.rows.withIndex()
        // Array variants
        is BooleanArray -> value.withIndex()
        is ByteArray -> value.withIndex()
        is CharArray -> value.withIndex()
        is DoubleArray -> value.withIndex()
        is FloatArray -> value.withIndex()
        is IntArray -> value.withIndex()
        is LongArray -> value.withIndex()
        is ShortArray -> value.withIndex()
        else ->
            error(
                "Unsupported value type: supported types are `Collection`, `Array` (including " +
                    "variants), and `Table`."
            )
    }
