package tech.ostack.kform.validations

import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import tech.ostack.kform.Validation
import tech.ostack.kform.ValidationContext
import tech.ostack.kform.ValidationIssue
import tech.ostack.kform.ValidationIssueSeverity
import tech.ostack.kform.validations.Matches.Companion.DEFAULT_CODE
import tech.ostack.kform.validations.MatchesEmail.Companion.DEFAULT_CODE
import tech.ostack.kform.validations.MatchesEmail.Companion.EMAIL_REGEX

/**
 * Validation that checks that a string matches a given [regex], when it is not empty.
 *
 * When the string is not empty and does not match [regex], then an issue is emitted with the
 * provided [code] (defaults to [DEFAULT_CODE]). This issue contains a `value` data property with
 * the value that was validated and a `pattern` data property with the pattern of the [regex] that
 * should have been matched.
 *
 * @property regex Regular expression that the string must match.
 * @property code Issue code to use when the string is not empty and does not match [regex].
 * @property severity Severity of the issue emitted when the string is not empty and does not match
 *   [regex].
 */
public open class Matches
@JvmOverloads
constructor(
    public val regex: Regex,
    public val code: String = DEFAULT_CODE,
    public val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
) : Validation<String>() {
    public constructor(
        pattern: String,
        code: String = DEFAULT_CODE,
        severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
    ) : this(Regex(pattern), code, severity)

    public constructor(
        pattern: String,
        option: RegexOption,
        code: String = DEFAULT_CODE,
        severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
    ) : this(Regex(pattern, option), code, severity)

    public constructor(
        pattern: String,
        options: Set<RegexOption>,
        code: String = DEFAULT_CODE,
        severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
    ) : this(Regex(pattern, options), code, severity)

    override fun toString(): String = "Matches(${regex.pattern})"

    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        if (value.isNotEmpty() && !value.matches(regex)) {
            emit(
                ValidationIssue(code, severity, mapOf("value" to value, "pattern" to regex.pattern))
            )
        }
    }

    public companion object {
        /** Issue code representing that the string is not empty and does not match [regex]. */
        public const val DEFAULT_CODE: String = "patternMismatch"
    }
}

/**
 * Validation that checks that a string matches an email according to [regex] (which defaults to
 * [EMAIL_REGEX]) when it is not empty.
 *
 * When the string is not empty and does not match an email according to [regex], then an issue is
 * emitted with the provided [code] (defaults to [DEFAULT_CODE]). This issue contains a `value` data
 * property with the value that was validated and a `pattern` data property with the email regular
 * expression that should have been matched.
 *
 * @param code Issue code to use when the string is not empty and does not match [regex].
 * @param severity Severity of the issue emitted when the string is not empty and does not match
 *   [regex].
 * @param regex Email regular expression that the string must match, defaults to [EMAIL_REGEX].
 */
public open class MatchesEmail
@JvmOverloads
constructor(
    code: String = DEFAULT_CODE,
    severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
    regex: Regex = EMAIL_REGEX,
) : Matches(regex, code, severity) {
    override fun toString(): String = "MatchesEmail"

    public companion object {
        /**
         * Email regular expression used by default to match an email. It matches emails according
         * to the W3C HTML5 spec (see:
         * https://html.spec.whatwg.org/multipage/input.html#valid-e-mail-address).
         */
        // [Matches] regex patterns may end up used in the `pattern` attribute of HTML inputs.
        // Regexes compiled from an HTML input's `pattern` use JS's `v` regex flag, which forbids
        // some unescaped special characters that we escape below (and that IJ complains about
        // without this suppression).
        @Suppress("RegExpRedundantEscape")
        public val EMAIL_REGEX: Regex =
            Regex(
                "^[a-zA-Z0-9.!#$%&'*+\\/=?^_`\\{\\|\\}~\\-]+@" +
                    "[a-zA-Z0-9](?:[a-zA-Z0-9\\-]{0,61}[a-zA-Z0-9])?" +
                    "(?:\\.[a-zA-Z0-9](?:[a-zA-Z0-9\\-]{0,61}[a-zA-Z0-9])?)*$"
            )

        /** Issue code representing that the string is not empty and does not match [regex]. */
        public const val DEFAULT_CODE: String = "emailPatternMismatch"
    }
}
