package tech.ostack.kform.validations

import kotlin.jvm.JvmName
import kotlin.jvm.JvmOverloads
import kotlin.jvm.JvmStatic
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import tech.ostack.kform.Computation
import tech.ostack.kform.ComputedValue
import tech.ostack.kform.ComputedValueContext
import tech.ostack.kform.DependencyInfo
import tech.ostack.kform.Observer
import tech.ostack.kform.StatefulComputedValue
import tech.ostack.kform.StatefulValidation
import tech.ostack.kform.Validation
import tech.ostack.kform.ValidationContext
import tech.ostack.kform.ValidationIssue
import tech.ostack.kform.ValidationIssueSeverity

/**
 * Validation that checks that a form value matches a given [computedValue].
 *
 * When the value being validated is different from the result of evaluating [computedValue], then
 * an issue is emitted with the provided [code] (defaults to [DEFAULT_CODE]). This issue contains a
 * `value` data property with the value that was validated and a `computedValue` data property with
 * the value that was computed.
 *
 * A validation of this kind is automatically added to declared
 * [computed schemas][tech.ostack.kform.schemas.ComputedSchema].
 */
public interface MatchesComputedValue<T> : Computation {
    /** Computed value that should be matched. */
    public val computedValue: ComputedValue<T>

    /** Issue code to use when the value differs from the [computedValue]. */
    public val code: String

    /** Severity of the issue emitted when the value differs from the [computedValue]. */
    public val severity: ValidationIssueSeverity

    public companion object {
        /** Default issue code representing that the value is different from the computed value. */
        public const val DEFAULT_CODE: String = "computedValueMismatch"

        /** Function used to create a [MatchesComputedValue] validation. */
        @JvmOverloads
        @JvmStatic
        @JvmName("create")
        public operator fun <T> invoke(
            computedValue: ComputedValue<T>,
            code: String = DEFAULT_CODE,
            severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
        ): Validation<T> =
            when (computedValue) {
                is StatefulComputedValue<*, *> ->
                    MatchesStatefulComputedValue(computedValue, code, severity)
                else -> MatchesStatelessComputedValue(computedValue, code, severity)
            }
    }
}

/** Validation that checks that a form value matches a given stateless [computedValue]. */
internal class MatchesStatelessComputedValue<T>(
    override val computedValue: ComputedValue<T>,
    override val code: String = MatchesComputedValue.DEFAULT_CODE,
    override val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
) : MatchesComputedValue<T>, Validation<T>() {
    override val dependsOnDescendants: Boolean = true

    init {
        (dependencies as MutableMap<String, DependencyInfo>).putAll(computedValue.dependencies)
        (externalContextDependencies as MutableSet<String>).addAll(
            computedValue.externalContextDependencies
        )
    }

    override fun toString(): String = "MatchesComputedValue($computedValue)"

    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        val computed = computedValue.run { computedValueContext().compute() }
        if (value != computed) {
            emit(
                ValidationIssue(
                    code,
                    severity,
                    mapOf("value" to "$value", "computedValue" to computed.toString()),
                )
            )
        }
    }
}

/** Validation that checks that a form value matches a given stateful [computedValue]. */
internal class MatchesStatefulComputedValue<T, TState>(
    override val computedValue: StatefulComputedValue<T, TState>,
    override val code: String = MatchesComputedValue.DEFAULT_CODE,
    override val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
) : MatchesComputedValue<T>, StatefulValidation<T, TState>() {
    override val dependsOnDescendants: Boolean = true

    init {
        (dependencies as MutableMap<String, DependencyInfo>).putAll(computedValue.dependencies)
        (externalContextDependencies as MutableSet<String>).addAll(
            computedValue.externalContextDependencies
        )
        (observers as MutableList<Observer<Any?, TState>>).addAll(computedValue.observers)
    }

    override fun toString(): String = "MatchesComputedValue($computedValue)"

    override suspend fun ValidationContext.initState(): TState =
        computedValue.run { computedValueContext().initState() }

    override suspend fun destroyState(state: TState): Unit = computedValue.destroyState(state)

    override fun ValidationContext.validateFromState(state: TState): Flow<ValidationIssue> = flow {
        val computed = computedValue.run { computedValueContext().computeFromState(state) }
        if (value != computed) {
            emit(ValidationIssue(code, severity, mapOf("computedValue" to computed.toString())))
        }
    }
}

/** Function which converts a validation context into a computed value context. */
private fun ValidationContext.computedValueContext() =
    ComputedValueContext(schema<Any?>(), path, schemaPath, dependenciesInfo, externalContexts)
