package tech.ostack.kform.validations

import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import tech.ostack.kform.Validation
import tech.ostack.kform.ValidationContext
import tech.ostack.kform.ValidationIssue
import tech.ostack.kform.ValidationIssueSeverity
import tech.ostack.kform.validations.Length.Companion.DEFAULT_CODE
import tech.ostack.kform.validations.MaxLength.Companion.DEFAULT_CODE
import tech.ostack.kform.validations.MinLength.Companion.DEFAULT_CODE
import tech.ostack.kform.validations.Required.Companion.DEFAULT_CODE

/**
 * Validation that checks that a string's length is exactly [requiredLength], when it is not empty.
 *
 * When the string being validated is not empty and its length is different from [requiredLength],
 * then an issue is emitted with the provided [code] (defaults to [DEFAULT_CODE]). This issue
 * contains a `length` data property with the length of the string that was validated and a
 * `requiredLength` data property with the value of [requiredLength].
 *
 * To also validate that a string is not empty, use [Required] (preferred) or [NotEmpty] (if the
 * value is nullable and `null` is an acceptable value) together with [Length].
 *
 * Example values accepted by schema `StringSchema(Length(4))`:
 * - `"abcd"`
 * - `""`
 *
 * Example values rejected by schema `StringSchema(Length(4))`:
 * - `"abc"`
 * - `"abcde"`
 *
 * @property requiredLength Required length for non-empty strings (must be >= 0).
 * @property code Issue code to use when the string is not empty and its length is different from
 *   [requiredLength].
 * @property severity Severity of the issue emitted when the string is not empty and its length is
 *   different from [requiredLength].
 */
public open class Length
@JvmOverloads
constructor(
    public val requiredLength: Int,
    public val code: String = DEFAULT_CODE,
    public val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
) : Validation<CharSequence>() {
    init {
        require(requiredLength >= 0) { "Provided length must be >= 0." }
    }

    override fun toString(): String = "Length($requiredLength)"

    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        if (value.isNotEmpty() && value.length != requiredLength) {
            emit(
                ValidationIssue(
                    code,
                    severity,
                    mapOf("length" to "${value.length}", "requiredLength" to "$requiredLength"),
                )
            )
        }
    }

    public companion object {
        /**
         * Default issue code representing that the string is not empty and its length is different
         * from [requiredLength].
         */
        public const val DEFAULT_CODE: String = "lengthMismatch"
    }
}

/**
 * Validation that checks that a string's length is at least a given [minLength], when it is not
 * empty.
 *
 * When the string being validated is not empty and its length is less than [minLength], then an
 * issue is emitted with the provided [code] (defaults to [DEFAULT_CODE]). This issue contains a
 * `length` data property with the length of the string that was validated and a `minLength` data
 * property with the value of [minLength].
 *
 * To also validate that a string is not empty, use [Required] (preferred) or [NotEmpty] (if the
 * value is nullable and `null` is an acceptable value) together with [MinLength].
 *
 * Example values accepted by schema `StringSchema(MinLength(4))`:
 * - `"abcd"`
 * - `"abcdefgh"`
 * - `""`
 *
 * Example values rejected by schema `StringSchema(MinLength(4))`:
 * - `"a"`
 * - `"abc"`
 *
 * @property minLength Minimum length required for non-empty strings (must be >= 0).
 * @property code Issue code to use when the string is not empty and its length is less than
 *   [minLength].
 * @property severity Severity of the issue emitted when the string is not empty and its length is
 *   less than [minLength].
 */
public open class MinLength
@JvmOverloads
constructor(
    public val minLength: Int,
    public val code: String = DEFAULT_CODE,
    public val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
) : Validation<CharSequence>() {
    init {
        require(minLength >= 0) { "Provided minimum length must be >= 0." }
    }

    override fun toString(): String = "MinLength($minLength)"

    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        if (value.isNotEmpty() && value.length < minLength) {
            emit(
                ValidationIssue(
                    code,
                    severity,
                    mapOf("length" to "${value.length}", "minLength" to "$minLength"),
                )
            )
        }
    }

    public companion object {
        /** Default issue code representing that the string's length is less than [minLength]. */
        public const val DEFAULT_CODE: String = "tooShort"
    }
}

/**
 * Validation that checks that a string's length is at most a given [maxLength].
 *
 * When the length of the string being validated is greater than [maxLength], then an issue is
 * emitted with the provided [code] (defaults to [DEFAULT_CODE]). This issue contains a `length`
 * data property with the length of the string that was validated and a `maxLength` data property
 * with the value of [maxLength].
 *
 * Example values accepted by schema `StringSchema(MaxLength(4))`:
 * - `"ab"`
 * - `"abcd"`
 * - `""`
 *
 * Example values rejected by schema `StringSchema(MaxLength(4))`:
 * - `"abcde"`
 * - `"abcdefgh"`
 *
 * @property maxLength Maximum length allowed (must be >= 0).
 * @property code Issue code to use when the string's length is greater than [maxLength].
 * @property severity Severity of the issue emitted when the string's length is greater than
 *   [maxLength].
 */
public open class MaxLength
@JvmOverloads
constructor(
    public val maxLength: Int,
    public val code: String = DEFAULT_CODE,
    public val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
) : Validation<CharSequence>() {
    init {
        require(maxLength >= 0) { "Provided maximum length must be >= 0." }
    }

    override fun toString(): String = "MaxLength($maxLength)"

    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        if (value.length > maxLength) {
            emit(
                ValidationIssue(
                    code,
                    severity,
                    mapOf("length" to "${value.length}", "maxLength" to "$maxLength"),
                )
            )
        }
    }

    public companion object {
        /** Default issue code representing that the string's length is greater than [maxLength]. */
        public const val DEFAULT_CODE: String = "tooLong"
    }
}
