@file:JvmName("TestUtils")
@file:JvmMultifileClass

package tech.ostack.kform.test

import kotlin.jvm.JvmMultifileClass
import kotlin.jvm.JvmName
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.toList
import tech.ostack.kform.ValidationIssue

/** Whether an [expectedIssue] matches the [actualIssue]. */
private fun issuesMatch(expectedIssue: ValidationIssue, actualIssue: ValidationIssue): Boolean =
    actualIssue.contains(expectedIssue)

/**
 * Asserts that the [expectedIssues] match the [actualIssues]. The order in which the issues appear
 * is not relevant.
 *
 * Matching issues are those where the actual issue [contains][ValidationIssue.contains] the
 * expected issue.
 */
@JvmOverloads
public fun assertMatchingIssues(
    expectedIssues: Iterable<ValidationIssue>,
    actualIssues: Iterable<ValidationIssue>,
    message: String? = null,
) {
    val expectedIssuesMap = expectedIssues.groupBy { it.code }
    val actualIssuesMap = actualIssues.groupBy { it.code }

    for (expectedIssue in expectedIssues) {
        val similarActualIssues = actualIssuesMap[expectedIssue.code]
        if (
            similarActualIssues == null ||
                !similarActualIssues.any { actualIssue -> issuesMatch(expectedIssue, actualIssue) }
        ) {
            throw AssertionError(
                messagePrefix(message) +
                    "Expected issue <$expectedIssue> does not match any actual issue.\n" +
                    "Expected issues: <$expectedIssues>, actual issues: <$actualIssues>."
            )
        }
    }
    for (actualIssue in actualIssues) {
        val similarExpectedIssues = expectedIssuesMap[actualIssue.code]
        if (
            similarExpectedIssues == null ||
                !similarExpectedIssues.any { expectedIssue ->
                    issuesMatch(expectedIssue, actualIssue)
                }
        ) {
            throw AssertionError(
                messagePrefix(message) +
                    "Actual issue <$actualIssue> does not match any expected issue.\n" +
                    "Expected issues: <$expectedIssues>, actual issues: <$actualIssues>."
            )
        }
    }
}

/**
 * Asserts that the [expectedIssues] match the [actualIssues]. The order in which the issues appear
 * is not relevant.
 *
 * Matching issues are those where the actual issue [contains][ValidationIssue.contains] the
 * expected issue.
 */
@JvmOverloads
public suspend fun assertMatchingIssues(
    expectedIssues: Iterable<ValidationIssue>,
    actualIssues: Flow<ValidationIssue>,
    message: String? = null,
): Unit = assertMatchingIssues(expectedIssues, actualIssues.toList(), message)

/**
 * Asserts that the [actualIssues] contains all matching [expectedIssues]. The order in which the
 * issues appear is not relevant.
 *
 * Matching issues are those where the actual issue [contains][ValidationIssue.contains] the
 * expected issue.
 */
@JvmOverloads
public fun assertContainsMatchingIssues(
    expectedIssues: Iterable<ValidationIssue>,
    actualIssues: Iterable<ValidationIssue>,
    message: String? = null,
) {
    val actualIssuesMap = actualIssues.groupBy { it.code }

    for (expectedIssue in expectedIssues) {
        val similarActualIssues = actualIssuesMap[expectedIssue.code]
        if (
            similarActualIssues == null ||
                !similarActualIssues.any { actualIssue -> issuesMatch(expectedIssue, actualIssue) }
        ) {
            throw AssertionError(
                messagePrefix(message) +
                    "Expected actual issues to contain expected issues.\n" +
                    "Expected issue <$expectedIssue> does not match any actual issue.\n" +
                    "Expected issues: <$expectedIssues>, actual issues: <$actualIssues>."
            )
        }
    }
}

/**
 * Asserts that the [actualIssues] contains all matching [expectedIssues]. The order in which the
 * issues appear is not relevant.
 *
 * Matching issues are those where the actual issue [contains][ValidationIssue.contains] the
 * expected issue.
 */
@JvmOverloads
public suspend fun assertContainsMatchingIssues(
    expectedIssues: Iterable<ValidationIssue>,
    actualIssues: Flow<ValidationIssue>,
    message: String? = null,
): Unit = assertContainsMatchingIssues(expectedIssues, actualIssues.toList(), message)

/**
 * Asserts that the [actualIssues] does **not** contain any matching issues in [forbiddenIssues].
 * The order in which the issues appear is not relevant.
 *
 * Matching issues are those where the actual issue [contains][ValidationIssue.contains] the
 * expected issue.
 */
@JvmOverloads
public fun assertNotContainsMatchingIssues(
    forbiddenIssues: Iterable<ValidationIssue>,
    actualIssues: Iterable<ValidationIssue>,
    message: String? = null,
) {
    val actualIssuesMap = actualIssues.groupBy { it.code }

    for (forbiddenIssue in forbiddenIssues) {
        val similarActualIssues = actualIssuesMap[forbiddenIssue.code]
        if (
            similarActualIssues != null &&
                similarActualIssues.any { actualIssue -> issuesMatch(forbiddenIssue, actualIssue) }
        ) {
            throw AssertionError(
                messagePrefix(message) +
                    "Expected actual issues to not contain forbidden issues.\n" +
                    "Forbidden issue <$forbiddenIssue> matches an actual issue.\n" +
                    "Forbidden issues: <$forbiddenIssues>, actual issues: <$actualIssues>."
            )
        }
    }
}

/**
 * Asserts that the [actualIssues] does **not** contain any matching issues in [forbiddenIssues].
 * The order in which the issues appear is not relevant.
 *
 * Matching issues are those where the actual issue [contains][ValidationIssue.contains] the
 * expected issue.
 */
@JvmOverloads
public suspend fun assertNotContainsMatchingIssues(
    forbiddenIssues: Iterable<ValidationIssue>,
    actualIssues: Flow<ValidationIssue>,
    message: String? = null,
): Unit = assertNotContainsMatchingIssues(forbiddenIssues, actualIssues.toList(), message)

/**
 * Asserts that [actualIssues] contains a matching [expectedIssue].
 *
 * Matching issues are those where the actual issue [contains][ValidationIssue.contains] the
 * expected issue.
 */
@JvmOverloads
public fun assertContainsMatchingIssue(
    expectedIssue: ValidationIssue,
    actualIssues: Iterable<ValidationIssue>,
    message: String? = null,
) {
    for (actualIssue in actualIssues) {
        if (issuesMatch(expectedIssue, actualIssue)) {
            return
        }
    }

    throw AssertionError(
        messagePrefix(message) +
            "Expected actual issues to contain expected issue.\n" +
            "Expected issue: <$expectedIssue>, actual issues: <$actualIssues>."
    )
}

/**
 * Asserts that [actualIssues] contains a matching [expectedIssue].
 *
 * Matching issues are those where the actual issue [contains][ValidationIssue.contains] the
 * expected issue.
 */
@JvmOverloads
public suspend fun assertContainsMatchingIssue(
    expectedIssue: ValidationIssue,
    actualIssues: Flow<ValidationIssue>,
    message: String? = null,
): Unit = assertContainsMatchingIssue(expectedIssue, actualIssues.toList(), message)

/**
 * Asserts that [actualIssues] does not contain a matching [forbiddenIssue].
 *
 * Matching issues are those where the actual issue [contains][ValidationIssue.contains] the
 * expected issue.
 */
@JvmOverloads
public fun assertNotContainsMatchingIssue(
    forbiddenIssue: ValidationIssue,
    actualIssues: Iterable<ValidationIssue>,
    message: String? = null,
) {
    for (actualIssue in actualIssues) {
        if (issuesMatch(forbiddenIssue, actualIssue)) {
            throw AssertionError(
                messagePrefix(message) +
                    "Expected actual issues to not contain forbidden issue.\n" +
                    "Forbidden issue: <$forbiddenIssue>, actual issues: <$actualIssues>."
            )
        }
    }
}

/**
 * Asserts that [actualIssues] does not contain a matching [forbiddenIssue].
 *
 * Matching issues are those where the actual issue [contains][ValidationIssue.contains] the
 * expected issue.
 */
@JvmOverloads
public suspend fun assertNotContainsMatchingIssue(
    forbiddenIssue: ValidationIssue,
    actualIssues: Flow<ValidationIssue>,
    message: String? = null,
): Unit = assertNotContainsMatchingIssue(forbiddenIssue, actualIssues.toList(), message)
