@file:JvmName("SchemaRestrictions")

package tech.ostack.kform.schemas.util

import kotlin.jvm.JvmName
import tech.ostack.kform.Validation
import tech.ostack.kform.datatypes.BigDecimal
import tech.ostack.kform.datatypes.File
import tech.ostack.kform.validations.*

/** Builds the common restrictions of a schema, given its validations. */
public fun <T> commonRestrictions(validations: Iterable<Validation<T>>): Map<String, Any?> =
    buildMap {
        if (validations.any { it is Required }) {
            put("required", true)
        }

        val allowedValuesSets =
            validations.filterIsInstance<OneOf<T>>().map { it.allowedValues } +
                validations.filterIsInstance<MustEqual<T>>().map { setOf(it.requiredValue) }
        if (allowedValuesSets.isNotEmpty()) {
            put(
                "allowedValues",
                allowedValuesSets
                    .reduce { intersectedAllowedValues, allowedValues ->
                        intersectedAllowedValues intersect allowedValues
                    }
                    .toTypedArray<Any?>(),
            )
        }

        val disallowedValues =
            validations.filterIsInstance<NotOneOf<T>>().flatMapTo(mutableSetOf()) {
                it.disallowedValues
            } + validations.filterIsInstance<MustNotEqual<T>>().map { it.forbiddenValue }
        if (disallowedValues.isNotEmpty()) {
            put("disallowedValues", disallowedValues.toTypedArray<Any?>())
        }
    }

/** Builds the comparable bounds restrictions of a schema, given its validations. */
public fun <T : Comparable<T>> comparableBoundsRestrictions(
    validations: Iterable<Validation<T>>,
    typeMin: T? = null,
    typeMax: T? = null,
): Map<String, T> = buildMap {
    val validationMin = validations.filterIsInstance<Min<T>>().maxOfOrNull { it.min }
    val min =
        if (typeMin != null && validationMin != null) maxOf(typeMin, validationMin)
        else typeMin ?: validationMin
    if (min != null) {
        put("min", min)
    }

    val validationMax = validations.filterIsInstance<Max<T>>().minOfOrNull { it.max }
    val max =
        if (typeMax != null && validationMax != null) minOf(typeMax, validationMax)
        else typeMax ?: validationMax
    if (max != null) {
        put("max", max)
    }

    val exclusiveMin =
        validations.filterIsInstance<ExclusiveMin<T>>().maxOfOrNull { it.exclusiveMin }
    if (exclusiveMin != null) {
        put("exclusiveMin", exclusiveMin)
    }

    val exclusiveMax =
        validations.filterIsInstance<ExclusiveMax<T>>().minOfOrNull { it.exclusiveMax }
    if (exclusiveMax != null) {
        put("exclusiveMax", exclusiveMax)
    }
}

/** Builds the size bounds restrictions of a schema, given its validations. */
public fun sizeBoundsRestrictions(validations: Iterable<Validation<*>>): Map<String, Int> =
    buildMap {
        val minSize =
            (validations.filterIsInstance<MinSize>() + validations.filterIsInstance<Size>())
                .maxOfOrNull { if (it is MinSize) it.minSize else (it as Size).requiredSize }
        if (minSize != null) {
            put("minSize", minSize)
        }

        val maxSize =
            (validations.filterIsInstance<MaxSize>() + validations.filterIsInstance<Size>())
                .minOfOrNull { if (it is MaxSize) it.maxSize else (it as Size).requiredSize }
        if (maxSize != null) {
            put("maxSize", maxSize)
        }
    }

/** Builds the size length restrictions of a schema, given its validations. */
public fun <T : CharSequence> lengthBoundsRestrictions(
    validations: Iterable<Validation<T>>
): Map<String, Int> = buildMap {
    val minLength =
        (validations.filterIsInstance<MinLength>() + validations.filterIsInstance<Length>())
            .maxOfOrNull { if (it is MinLength) it.minLength else (it as Length).requiredLength }
    if (minLength != null) {
        put("minLength", minLength)
    }

    val maxLength =
        (validations.filterIsInstance<MaxLength>() + validations.filterIsInstance<Length>())
            .minOfOrNull { if (it is MaxLength) it.maxLength else (it as Length).requiredLength }
    if (maxLength != null) {
        put("maxLength", maxLength)
    }
}

/** Builds the pattern restrictions of a schema, given its validations. */
public fun patternRestrictions(validations: Iterable<Validation<String>>): Map<String, Any> =
    buildMap {
        val matchesValidations = validations.filterIsInstance<Matches>()
        if (matchesValidations.isNotEmpty()) {
            put(
                "pattern",
                if (matchesValidations.size == 1) matchesValidations.first().regex.pattern
                else matchesValidations.joinToString("|") { "(${it.regex.pattern})" },
            )
        }
    }

/** Builds the scale restrictions of a schema, given its validations. */
public fun scaleRestriction(validations: Iterable<Validation<BigDecimal>>): Map<String, Any> =
    buildMap {
        val scales = validations.filterIsInstance<Scale>().map { it.requiredScale }.toSet()
        if (scales.size == 1) {
            put("scale", scales.first())
        }
    }

/** Builds the accepted file types restrictions of a schema, given its validations. */
public fun acceptedFileTypesRestrictions(
    validations: Iterable<Validation<File>>
): Map<String, Any> = buildMap {
    val acceptedFileTypesValidations = validations.filterIsInstance<Accepts>()
    if (acceptedFileTypesValidations.isNotEmpty()) {
        // FIXME: This is incorrect, if one [Accepts] validation allows image/* and another
        //  image/jpeg, the intersection should be image/jpeg, rather than the empty set
        put(
            "acceptedFileTypes",
            acceptedFileTypesValidations
                .map { it.acceptedFileTypes }
                .reduce { intersectedAcceptedFileTypes, acceptedFileTypes ->
                    intersectedAcceptedFileTypes intersect acceptedFileTypes
                }
                .toTypedArray<Any?>(),
        )
    }
}
