package tech.ostack.kform.schemas

import kotlin.jvm.JvmName
import kotlin.jvm.JvmOverloads
import kotlin.jvm.JvmStatic
import kotlin.reflect.KType
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.emitAll
import kotlinx.coroutines.flow.emptyFlow
import kotlinx.coroutines.flow.flow
import tech.ostack.kform.*
import tech.ostack.kform.schemas.util.commonRestrictions
import tech.ostack.kform.validations.*

/** Schema representing nullable values. */
public interface NullableSchema<T> : Schema<T?> {
    public val innerSchema: Schema<T>

    public companion object {
        /** Function that builds a schema representing nullable values of type [T]. */
        @JvmOverloads
        @JvmStatic
        @JvmName("create")
        @Suppress("UNCHECKED_CAST")
        public operator fun <T> invoke(
            validations: Iterable<Validation<T?>> = emptyList(),
            initialValue: T? = null,
            @BuilderInference builder: () -> Schema<T>,
        ): NullableSchema<T> =
            when (val schema = builder()) {
                is CollectionSchema<*, *> -> {
                    schema as CollectionSchema<T, Any?>
                    NullableCollectionSchema(schema, validations, initialValue)
                }
                is ParentSchema<*> -> {
                    schema as ParentSchema<T>
                    NullableParentSchema<T, Any?, ParentSchema<T>>(
                        schema,
                        validations,
                        initialValue,
                    )
                }
                else -> NullableSimpleSchema(schema, validations, initialValue)
            }

        /** Function that builds a schema representing nullable values of type [T]. */
        @JvmOverloads
        @JvmStatic
        @JvmName("create")
        public operator fun <T> invoke(
            vararg validations: Validation<T?>,
            initialValue: T? = null,
            @BuilderInference builder: () -> Schema<T>,
        ): NullableSchema<T> = NullableSchema(validations.asIterable(), initialValue, builder)
    }
}

/**
 * Schema representing nullable simple values of type [T] represented by schemas of type [TSchema].
 */
internal open class NullableSimpleSchema<T, TSchema : Schema<T>>(
    final override val innerSchema: TSchema,
    validations: Iterable<Validation<T?>> = emptyList(),
    override val initialValue: T? = null,
) : Schema<T?>, NullableSchema<T> {
    init {
        require(innerSchema !is NullableSchema<*>) {
            "Cannot create a nullable schema of a nullable schema."
        }
    }

    override val typeInfo: TypeInfo =
        innerSchema.typeInfo.copy(
            nullable = true,
            restrictions =
                buildMap {
                    putAll(innerSchema.typeInfo.restrictions)
                    // Merge allowed and disallowed values of this schema with the inner schema's,
                    // overwriting the restrictions added above, if necessary
                    putAll(commonRestrictions(validations + innerSchema.validations))
                    // If this schema doesn't have a [Required] validation, then `null` values may
                    // be allowed, in which case the schema shouldn't be marked as required, even if
                    // the inner schema is
                    if (validations.none { it is Required }) {
                        remove("required")
                    }
                },
        )

    /** Wrapper over [validation] in order to handle `null` values. */
    class WrappedNullableValidation<T>(val validation: Validation<T>) : Validation<T?>() {
        override val dependencies: Map<String, DependencyInfo> = validation.dependencies
        override val dependsOnDescendants: Boolean = validation.dependsOnDescendants
        override val externalContextDependencies: Set<String> =
            validation.externalContextDependencies

        override fun toString(): String = "$validation?"

        override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
            if (value != null) {
                validation.run { emitAll(validate()) }
            }
        }
    }

    /**
     * Class holding the state of a stateful validation. Used to be able to change the state during
     * `validateFromState`.
     */
    // NOTE: We wrap the state in order to be able to update it during [validateFromState]; this is
    //  safe because the manager keeps a lock on the state during validation, so no concurrent
    //  updating of state can occur
    data class WrappedState<TState>(var innerState: TState?, var isKnown: Boolean) {
        fun setKnownState(state: TState) {
            innerState = state
            isKnown = true
        }

        fun setUnknownState() {
            innerState = null
            isKnown = false
        }
    }

    /** Wrapper over a [stateful validation][validation] in order to handle `null` values. */
    class WrappedNullableStatefulValidation<T, TState>(
        private val nullableSchema: Schema<T?>,
        private val schema: Schema<T>,
        val validation: StatefulValidation<T, TState>,
    ) : StatefulValidation<T?, WrappedState<TState>>() {
        override val dependencies: Map<String, DependencyInfo> = validation.dependencies
        override val dependsOnDescendants: Boolean = validation.dependsOnDescendants
        override val externalContextDependencies: Set<String> =
            validation.externalContextDependencies

        override val observers: List<Observer<Any?, WrappedState<TState>>> =
            validation.observers.map { observer ->
                Observer(observer.toObserve, updateStateFn(observer))
            }

        override fun toString(): String = "$validation?"

        override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
            if (value != null) {
                validation.run { emitAll(validate()) }
            }
        }

        override suspend fun ValidationContext.initState(): WrappedState<TState> =
            if (value != null) WrappedState(validation.run { initState() }, true)
            else WrappedState(null, false)

        private fun updateStateFn(
            observer: Observer<Any?, TState>
        ): UpdateStateFn<Any?, WrappedState<TState>> = fn@{ state, event ->
            // Do nothing if we don't know the state of the validation (it will be set by
            // [validateFromState] when appropriate).
            if (!state.isKnown) {
                return@fn state
            }

            // In case the stateful validation is observing the path of the value it is
            // validating,
            // we need to replace the event's schema and make sure that it never sees `null`.
            @Suppress("UNCHECKED_CAST")
            val replacedEvent =
                if (event.schema !== nullableSchema) event
                else
                    when (event) {
                        // This should never happen (the manager should never emit init/destroy
                        // events for the value being validated).
                        is ValueEvent.Init,
                        is ValueEvent.Destroy ->
                            error(
                                "Nullable stateful validation received an [init]/[destroy] " +
                                    "event for the value being validated."
                            )
                        is ValueEvent.Change<*> -> {
                            event as ValueEvent.Change<T?>
                            if (event.oldValue == null || event.value == null) {
                                return@fn WrappedState(null, false)
                            }
                            ValueEvent.Change(event.oldValue, event.value, event.path, schema)
                        }
                        is ValueEvent.Add<*, *> ->
                            ValueEvent.Add(
                                event.value as T,
                                event.addedValue,
                                event.id,
                                event.path,
                                schema as CollectionSchema<T, Any?>,
                            )
                        is ValueEvent.Remove<*, *> ->
                            ValueEvent.Remove(
                                event.value as T,
                                event.removedValue,
                                event.id,
                                event.path,
                                schema as CollectionSchema<T, Any?>,
                            )
                    }

            @Suppress("UNCHECKED_CAST")
            return@fn WrappedState(
                observer.updateState(state.innerState as TState, replacedEvent as ValueEvent<Any?>),
                true,
            )
        }

        @Suppress("UNCHECKED_CAST")
        override suspend fun destroyState(state: WrappedState<TState>): Unit =
            if (state.isKnown) validation.destroyState(state.innerState as TState) else Unit

        override fun ValidationContext.validateFromState(
            state: WrappedState<TState>
        ): Flow<ValidationIssue> = flow {
            if (value != null) {
                if (!state.isKnown) {
                    state.setKnownState(validation.run { initState() })
                }
                @Suppress("UNCHECKED_CAST")
                validation.run { emitAll(validateFromState(state.innerState as TState)) }
            } else if (state.isKnown) {
                state.setUnknownState()
            }
        }
    }

    override val validations: List<Validation<T?>> =
        validations +
            buildList {
                for (validation in innerSchema.validations) {
                    // Wrap the inner schema validations
                    add(
                        if (validation is StatefulValidation<*, *>) {
                            @Suppress("UNCHECKED_CAST") (validation as StatefulValidation<T, Any?>)
                            WrappedNullableStatefulValidation(
                                this@NullableSimpleSchema,
                                innerSchema,
                                validation,
                            )
                        } else WrappedNullableValidation(validation)
                    )
                }
            }

    override suspend fun clone(value: T?): T? =
        if (value == null) null else innerSchema.clone(value)

    override fun assignableTo(type: KType): Boolean =
        type.isMarkedNullable && innerSchema.assignableTo(type)

    /**
     * Changes [event] when its path is [path] (the path of this schema) so that it references this
     * (nullable) schema instead of [innerSchema]. Returns [event] when nothing needs to be changed
     * or the new (replaced) event otherwise.
     *
     * @param replaceWithChange `Init` and `Destroy` events are replaced with `Change` to/from
     *   `null`.
     */
    protected open fun replaceEvent(
        path: AbsolutePath,
        event: ValueEvent<*>,
        replaceWithChange: Boolean,
    ): ValueEvent<*> {
        if (path != event.path) {
            return event
        }
        @Suppress("UNCHECKED_CAST")
        return when (event) {
            is ValueEvent.Init<*> ->
                if (replaceWithChange) ValueEvent.Change(null, event.value as T, path, this)
                else ValueEvent.Init(event.value as T, path, this)
            is ValueEvent.Change<*> ->
                ValueEvent.Change(event.oldValue as T, event.value as T, path, this)
            is ValueEvent.Destroy<*> ->
                if (replaceWithChange) ValueEvent.Change(event.oldValue as T, null, path, this)
                else ValueEvent.Destroy(event.oldValue as T, path, this)
            else -> event
        }
    }

    /**
     * Runs [fn] with a custom events bus where all events emitted to [eventsBus] are mapped through
     * [replaceEvent]: this makes sure that no schema event whose path is [path] (the path of this
     * schema) references a schema that isn't this one.
     *
     * @param replaceWithChange `Init` and `Destroy` events are replaced with `Change` to/from
     *   `null`.
     */
    protected suspend inline fun withReplacedEvents(
        path: AbsolutePath,
        eventsBus: SchemaEventsBus,
        replaceWithChange: Boolean,
        crossinline fn: suspend (eventsBus: SchemaEventsBus) -> Unit,
    ) =
        fn(
            object : SchemaEventsBus {
                override suspend fun emit(event: ValueEvent<*>) {
                    eventsBus.emit(replaceEvent(path, event, replaceWithChange))
                }
            }
        )

    override suspend fun init(
        path: AbsolutePath,
        fromValue: Any?,
        eventsBus: SchemaEventsBus,
        setValue: suspend (value: T?) -> Unit,
    ) {
        if (fromValue == null) {
            setValue(null)
            eventsBus.emit(ValueEvent.Init(fromValue, path, this))
        } else {
            withReplacedEvents(path, eventsBus, false) { replacedEventsBus ->
                innerSchema.init(path, fromValue, replacedEventsBus, setValue)
            }
        }
    }

    override suspend fun change(
        path: AbsolutePath,
        value: T?,
        intoValue: Any?,
        eventsBus: SchemaEventsBus,
        setValue: suspend (value: T?) -> Unit,
    ) {
        if (value == null && intoValue == null) {
            setValue(null)
        } else {
            withReplacedEvents(path, eventsBus, true) { replacedEventsBus ->
                if (value == null) {
                    innerSchema.init(path, intoValue, replacedEventsBus, setValue)
                } else if (intoValue == null) {
                    innerSchema.destroy(path, value, replacedEventsBus) { setValue(null) }
                } else {
                    innerSchema.change(path, value, intoValue, replacedEventsBus, setValue)
                }
            }
        }
    }

    override suspend fun destroy(
        path: AbsolutePath,
        value: T?,
        eventsBus: SchemaEventsBus,
        removeValue: suspend (value: T?) -> Unit,
    ) {
        if (value == null) {
            removeValue(null)
            eventsBus.emit(ValueEvent.Destroy(null, path, this))
        } else {
            withReplacedEvents(path, eventsBus, false) { replacedEventsBus ->
                innerSchema.destroy(path, value, replacedEventsBus, removeValue)
            }
        }
    }
}

/**
 * Schema representing nullable parent values of type [T] and schema [TSchema] with children of type
 * [TChildren].
 */
internal open class NullableParentSchema<T, TChildren, TSchema : ParentSchema<T>>(
    schema: TSchema,
    validations: Iterable<Validation<T?>> = emptyList(),
    initialValue: T? = null,
) : NullableSimpleSchema<T, TSchema>(schema, validations, initialValue), ParentSchema<T?> {
    override val supportsConcurrentSets: Boolean = innerSchema.supportsConcurrentSets

    override fun isValidChildSchemaFragment(fragment: AbsolutePathFragment): Boolean =
        innerSchema.isValidChildSchemaFragment(fragment)

    @Suppress("UNCHECKED_CAST")
    override fun childrenSchemas(
        path: AbsolutePath,
        queriedPath: AbsolutePath,
        fragment: AbsolutePathFragment,
    ): Sequence<SchemaInfo<TChildren>> =
        innerSchema.childrenSchemas(path, queriedPath, fragment) as Sequence<SchemaInfo<TChildren>>

    override suspend fun isValidChildFragment(value: T?, fragment: AbsolutePathFragment): Boolean =
        value != null && innerSchema.isValidChildFragment(value, fragment)

    @Suppress("UNCHECKED_CAST")
    override fun children(
        path: AbsolutePath,
        schemaPath: AbsolutePath,
        value: T?,
        fragment: AbsolutePathFragment,
    ): Flow<ValueInfo<TChildren>> =
        if (value != null)
            innerSchema.children(path, schemaPath, value, fragment) as Flow<ValueInfo<TChildren>>
        else emptyFlow()

    override suspend fun isValidSetFragment(value: T?, fragment: AbsolutePathFragment): Boolean =
        value != null && innerSchema.isValidSetFragment(value, fragment)

    override suspend fun set(
        path: AbsolutePath,
        value: T?,
        fragment: AbsolutePathFragment,
        childValue: Any?,
        eventsBus: SchemaEventsBus,
    ) {
        if (value == null) {
            throw AtPathException(path, "Cannot set child of null.")
        }
        withReplacedEvents(path, eventsBus, false) { replacedEventsBus ->
            innerSchema.set(path, value, fragment, childValue, replacedEventsBus)
        }
    }

    override fun childrenStatesContainer(): ParentState = innerSchema.childrenStatesContainer()
}

/**
 * Schema representing nullable collections of type [T] and schema [TSchema] with children of type
 * [TChildren].
 */
internal open class NullableCollectionSchema<
    T,
    TChildren,
    TSchema : CollectionSchema<T, TChildren>,
>(schema: TSchema, validations: Iterable<Validation<T?>> = emptyList(), initialValue: T? = null) :
    NullableParentSchema<T, TChildren, TSchema>(schema, validations, initialValue),
    CollectionSchema<T?, TChildren> {
    override val supportsConcurrentRemoves: Boolean = innerSchema.supportsConcurrentRemoves

    // Replace additional actions (`Add` and `Remove`)
    override fun replaceEvent(
        path: AbsolutePath,
        event: ValueEvent<*>,
        replaceWithChange: Boolean,
    ): ValueEvent<*> {
        if (path != event.path) {
            return event
        }
        @Suppress("UNCHECKED_CAST")
        return when (event) {
            is ValueEvent.Add<*, *> ->
                ValueEvent.Add(
                    event.value as T,
                    event.addedValue as TChildren,
                    event.id,
                    path,
                    this,
                )
            is ValueEvent.Remove<*, *> ->
                ValueEvent.Remove(
                    event.value as T,
                    event.removedValue as TChildren,
                    event.id,
                    path,
                    this,
                )
            else -> super.replaceEvent(path, event, replaceWithChange)
        }
    }

    override suspend fun isValidRemoveFragment(value: T?, fragment: AbsolutePathFragment): Boolean =
        value != null && innerSchema.isValidRemoveFragment(value, fragment)

    override suspend fun remove(
        path: AbsolutePath,
        value: T?,
        fragment: AbsolutePathFragment,
        eventsBus: SchemaEventsBus,
    ) {
        if (value == null) {
            throw AtPathException(path, "Cannot remove children from null.")
        }
        withReplacedEvents(path, eventsBus, false) { replacedEventsBus ->
            innerSchema.remove(path, value, fragment, replacedEventsBus)
        }
    }

    override fun childrenStatesContainer(): CollectionState = innerSchema.childrenStatesContainer()
}
