package tech.ostack.kform.schemas

import kotlin.reflect.KClass
import kotlin.reflect.KType
import kotlinx.datetime.LocalDateTime
import tech.ostack.kform.TypeInfo
import tech.ostack.kform.Validation
import tech.ostack.kform.schemas.util.commonRestrictions
import tech.ostack.kform.schemas.util.comparableBoundsRestrictions

/**
 * Schema representing values of type [LocalDateTime] defined by
 * [KotlinX DateTime](https://github.com/Kotlin/kotlinx-datetime).
 */
public open class LocalDateTimeSchema(
    validations: Iterable<Validation<LocalDateTime>> = emptyList(),
    override val initialValue: LocalDateTime = DEFAULT_INITIAL_VALUE,
) : AbstractSimpleSchema<LocalDateTime>(validations) {
    public constructor(
        vararg validations: Validation<LocalDateTime>,
        initialValue: LocalDateTime = DEFAULT_INITIAL_VALUE,
    ) : this(validations.asIterable(), initialValue)

    override val typeInfo: TypeInfo =
        TypeInfo(
            LocalDateTime::class,
            restrictions =
                commonRestrictions(validations) + comparableBoundsRestrictions(validations),
        )

    override fun assignableTo(type: KType): Boolean =
        (type.classifier as? KClass<*>)?.isInstance(DEFAULT_INITIAL_VALUE) == true

    override suspend fun fromAny(value: Any?): LocalDateTime =
        when (value) {
            is LocalDateTime -> value
            is String -> LocalDateTime.parse(value)
            else ->
                throw IllegalArgumentException("Cannot convert value '$value' to LocalDateTime.")
        }

    public companion object {
        /** Default initial value (epoch in UTC). */
        public val DEFAULT_INITIAL_VALUE: LocalDateTime = LocalDateTime(1970, 1, 1, 0, 0)
    }
}
