package tech.ostack.kform.schemas

import kotlin.reflect.KClass
import kotlin.reflect.KType
import kotlinx.datetime.LocalDate
import tech.ostack.kform.TypeInfo
import tech.ostack.kform.Validation
import tech.ostack.kform.schemas.util.commonRestrictions
import tech.ostack.kform.schemas.util.comparableBoundsRestrictions

/**
 * Schema representing values of type [LocalDate] defined by
 * [KotlinX DateTime](https://github.com/Kotlin/kotlinx-datetime).
 */
public open class LocalDateSchema(
    validations: Iterable<Validation<LocalDate>> = emptyList(),
    override val initialValue: LocalDate = DEFAULT_INITIAL_VALUE,
) : AbstractSimpleSchema<LocalDate>(validations) {
    public constructor(
        vararg validations: Validation<LocalDate>,
        initialValue: LocalDate = DEFAULT_INITIAL_VALUE,
    ) : this(validations.asIterable(), initialValue)

    override val typeInfo: TypeInfo =
        TypeInfo(
            LocalDate::class,
            restrictions =
                commonRestrictions(validations) + comparableBoundsRestrictions(validations),
        )

    override fun assignableTo(type: KType): Boolean =
        (type.classifier as? KClass<*>)?.isInstance(DEFAULT_INITIAL_VALUE) == true

    override suspend fun fromAny(value: Any?): LocalDate =
        when (value) {
            is LocalDate -> value
            is String -> LocalDate.parse(value)
            else -> throw IllegalArgumentException("Cannot convert value '$value' to LocalDate.")
        }

    public companion object {
        /** Default initial value (epoch in UTC). */
        public val DEFAULT_INITIAL_VALUE: LocalDate = LocalDate(1970, 1, 1)
    }
}
