package tech.ostack.kform.schemas

import kotlin.math.min
import kotlin.reflect.KClass
import kotlin.reflect.KType
import kotlinx.coroutines.currentCoroutineContext
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.isActive
import tech.ostack.kform.*
import tech.ostack.kform.datatypes.Table
import tech.ostack.kform.schemas.util.commonRestrictions
import tech.ostack.kform.schemas.util.sizeBoundsRestrictions

/** Schema representing lists of values of type [T] with a schema of type [TSchema]. */
public open class ListSchema<T, TSchema : Schema<T>>(
    elementsSchema: TSchema,
    validations: Iterable<Validation<List<T>>> = emptyList(),
    override val initialValue: List<T> = emptyList(),
) : AbstractCollectionSchema<List<T>, T, TSchema>(elementsSchema) {
    override val typeInfo: TypeInfo =
        TypeInfo(
            List::class,
            arguments = listOf(elementsSchema.typeInfo),
            restrictions = commonRestrictions(validations) + sizeBoundsRestrictions(validations),
        )

    override val validations: List<Validation<List<T>>> = validations.toList()

    public constructor(
        validations: Iterable<Validation<List<T>>> = emptyList(),
        initialValue: List<T> = emptyList(),
        @BuilderInference builder: () -> TSchema,
    ) : this(builder(), validations, initialValue)

    public constructor(
        vararg validations: Validation<List<T>>,
        initialValue: List<T> = emptyList(),
        @BuilderInference builder: () -> TSchema,
    ) : this(builder(), validations.asIterable(), initialValue)

    override suspend fun clone(value: List<T>): List<T> =
        value.map { el -> elementsSchema.clone(el) }

    override fun assignableTo(type: KType): Boolean =
        (type.classifier as? KClass<*>)?.isInstance(emptyList<T>()) == true &&
            (if (type.classifier == List::class)
                type.arguments[0].type == null ||
                    elementsSchema.assignableTo(type.arguments[0].type!!)
            else true)

    /** Whether the provided id fragment can be converted to an integer. */
    private fun isValidIndexId(fragment: AbsolutePathFragment.Id): Boolean =
        fragment.id.matches(INDEX_REGEX)

    override fun isValidChildSchemaFragment(fragment: AbsolutePathFragment): Boolean =
        fragment is AbsolutePathFragment.CollectionEnd ||
            (fragment is AbsolutePathFragment.Id && isValidIndexId(fragment))

    override suspend fun isValidChildFragment(
        value: List<T>,
        fragment: AbsolutePathFragment,
    ): Boolean =
        (fragment is AbsolutePathFragment.Id &&
            isValidIndexId(fragment) &&
            fragment.id.toInt() < value.size)

    override fun children(
        path: AbsolutePath,
        schemaPath: AbsolutePath,
        value: List<T>,
        fragment: AbsolutePathFragment,
    ): Flow<ValueInfo<T>> = flow {
        if (fragment is AbsolutePathFragment.Wildcard) {
            for ((i, elem) in value.withIndex()) {
                emit(
                    ValueInfo(
                        elem,
                        elementsSchema,
                        path.append(AbsolutePathFragment.Id(i)),
                        schemaPath.append(AbsolutePathFragment.Wildcard),
                    )
                )
            }
        } else {
            fragment as AbsolutePathFragment.Id
            val idx = fragment.id.toInt()
            if (idx < value.size) { // `idx == value.size` is considered valid
                emit(
                    ValueInfo(
                        value[idx],
                        elementsSchema,
                        path.append(fragment),
                        schemaPath.append(AbsolutePathFragment.Wildcard),
                    )
                )
            }
        }
    }

    @Suppress("UNCHECKED_CAST")
    private fun fromAny(value: Any?): List<T> =
        when (value) {
            is List<*> -> value as List<T>
            is Collection<*> -> (value as Collection<T>).toList()
            is Table<*> -> (value as Table<T>).values.toList()
            is Array<*> -> (value as Array<T>).toList()
            else -> throw IllegalArgumentException("Cannot convert value '$value' to List.")
        }

    override suspend fun init(
        path: AbsolutePath,
        fromValue: Any?,
        eventsBus: SchemaEventsBus,
        setValue: suspend (value: List<T>) -> Unit,
    ) {
        val fromList = fromAny(fromValue)
        val newValue = ArrayList<T>(fromList.size)
        for ((i, elem) in fromList.withIndex()) {
            if (!currentCoroutineContext().isActive) break // Don't do more work than necessary
            elementsSchema.init(path.append(AbsolutePathFragment.Id(i)), elem, eventsBus) {
                newValue += it
            }
        }
        setValue(newValue)
        eventsBus.emit(ValueEvent.Init(newValue, path, this))
    }

    override suspend fun change(
        path: AbsolutePath,
        value: List<T>,
        intoValue: Any?,
        eventsBus: SchemaEventsBus,
        setValue: suspend (value: List<T>) -> Unit,
    ) {
        val intoList = fromAny(intoValue)
        value as ArrayList<T>
        val curSize = value.size
        val newSize = intoList.size
        for (i in 0 until min(curSize, newSize)) {
            if (!currentCoroutineContext().isActive) break // Don't do more work than necessary
            elementsSchema.change(
                path.append(AbsolutePathFragment.Id(i)),
                value[i],
                intoList[i],
                eventsBus,
            ) {
                value[i] = it
            }
        }
        if (curSize > newSize) {
            for (i in curSize - 1 downTo newSize) {
                if (!currentCoroutineContext().isActive) break // Don't do more work than necessary
                val id = AbsolutePathFragment.Id(i)
                val oldChild = value[i]
                elementsSchema.destroy(path.append(id), oldChild, eventsBus) { value.removeLast() }
                eventsBus.emit(ValueEvent.Remove(value, oldChild, id, path, this))
            }
        } else if (curSize < newSize) {
            value.ensureCapacity(newSize)
            for (i in curSize until newSize) {
                if (!currentCoroutineContext().isActive) break // Don't do more work than necessary
                val id = AbsolutePathFragment.Id(i)
                elementsSchema.init(path.append(id), intoList[i], eventsBus) { value += it }
                eventsBus.emit(ValueEvent.Add(value, value[i], id, path, this))
            }
        }
        setValue(value)
    }

    override suspend fun destroy(
        path: AbsolutePath,
        value: List<T>,
        eventsBus: SchemaEventsBus,
        removeValue: suspend (value: List<T>) -> Unit,
    ) {
        removeValue(value)
        eventsBus.emit(ValueEvent.Destroy(value, path, this))
        for (i in value.lastIndex downTo 0) {
            elementsSchema.destroy(path.append(AbsolutePathFragment.Id(i)), value[i], eventsBus) {}
        }
    }

    override suspend fun isValidSetFragment(
        value: List<T>,
        fragment: AbsolutePathFragment,
    ): Boolean =
        fragment is AbsolutePathFragment.CollectionEnd ||
            (fragment is AbsolutePathFragment.Id &&
                isValidIndexId(fragment) &&
                fragment.id.toInt() < value.size)

    override suspend fun set(
        path: AbsolutePath,
        value: List<T>,
        fragment: AbsolutePathFragment,
        childValue: Any?,
        eventsBus: SchemaEventsBus,
    ) {
        value as ArrayList<T>
        if (fragment is AbsolutePathFragment.Wildcard) {
            for ((idx, elem) in value.withIndex()) {
                elementsSchema.change(
                    path.append(AbsolutePathFragment.Id(idx)),
                    elem,
                    childValue,
                    eventsBus,
                ) {
                    value[idx] = it
                }
            }
        } else {
            val size = value.size
            val id =
                if (fragment is AbsolutePathFragment.CollectionEnd) AbsolutePathFragment.Id(size)
                else fragment
            val idx = (id as AbsolutePathFragment.Id).id.toInt()
            if (idx == size) {
                elementsSchema.init(path.append(id), childValue, eventsBus) { value += it }
                eventsBus.emit(ValueEvent.Add(value, value[idx], id, path, this))
            } else {
                elementsSchema.change(path.append(id), value[idx], childValue, eventsBus) {
                    value[idx] = it
                }
            }
        }
    }

    override suspend fun isValidRemoveFragment(
        value: List<T>,
        fragment: AbsolutePathFragment,
    ): Boolean =
        (fragment is AbsolutePathFragment.Id &&
            isValidIndexId(fragment) &&
            fragment.id.toInt() < value.size)

    override suspend fun remove(
        path: AbsolutePath,
        value: List<T>,
        fragment: AbsolutePathFragment,
        eventsBus: SchemaEventsBus,
    ) {
        value as ArrayList<T>
        if (fragment is AbsolutePathFragment.Wildcard) {
            for (i in value.lastIndex downTo 0) {
                val id = AbsolutePathFragment.Id(i)
                val oldChild = value.last()
                elementsSchema.destroy(path.append(id), oldChild, eventsBus) { value.removeLast() }
                eventsBus.emit(ValueEvent.Remove(value, oldChild, id, path, this))
            }
        } else {
            val idx = (fragment as AbsolutePathFragment.Id).id.toInt()
            for (i in idx until value.lastIndex) {
                elementsSchema.change(
                    path.append(AbsolutePathFragment.Id(i)),
                    value[i],
                    value[i + 1],
                    eventsBus,
                ) {
                    value[i] = it
                }
            }
            val id = AbsolutePathFragment.Id(value.lastIndex)
            val oldChild = value.last()
            elementsSchema.destroy(path.append(id), oldChild, eventsBus) { value.removeLast() }
            eventsBus.emit(ValueEvent.Remove(value, oldChild, id, path, this))
        }
    }

    override fun childrenStatesContainer(): CollectionState = ListState(elementsSchema)

    public companion object {
        /** Regex used to determine if a certain identifier can be converted to an index. */
        private val INDEX_REGEX = Regex("^(0|[1-9]\\d*)$")
    }
}

/** Class responsible for holding the states of the children of a list. */
public class ListState(private val elementsSchema: Schema<*>) : CollectionState {
    /** List containing the state of each element of the list. */
    private val childrenStatesList = mutableListOf<State?>()

    override fun childrenStates(
        path: AbsolutePath,
        fragment: AbsolutePathFragment,
    ): Sequence<StateInfo<*>> = sequence {
        if (fragment is AbsolutePathFragment.Wildcard) {
            for ((idx, state) in childrenStatesList.withIndex()) {
                yield(StateInfo(state, elementsSchema, path + AbsolutePathFragment.Id(idx)))
            }
        } else {
            fragment as AbsolutePathFragment.Id
            val idx = fragment.id.toInt()
            if (idx < childrenStatesList.size) {
                yield(StateInfo(childrenStatesList[idx], elementsSchema, path + fragment))
            }
        }
    }

    override fun hasState(fragment: AbsolutePathFragment.Id): Boolean =
        fragment.id.toInt() < childrenStatesList.size

    override fun setState(fragment: AbsolutePathFragment.Id, state: State?) {
        val idx = fragment.id.toInt()
        if (idx < childrenStatesList.size) {
            childrenStatesList[idx] = state
        } else if (idx == childrenStatesList.size) {
            childrenStatesList.add(state)
        }
    }

    override fun removeState(fragment: AbsolutePathFragment.Id) {
        childrenStatesList.removeAt(fragment.id.toInt())
    }
}
