package tech.ostack.kform.schemas

import kotlin.enums.enumEntries
import kotlin.jvm.JvmName
import kotlin.jvm.JvmOverloads
import kotlin.jvm.JvmStatic
import kotlin.reflect.KClass
import kotlin.reflect.KType
import tech.ostack.kform.TypeInfo
import tech.ostack.kform.Validation
import tech.ostack.kform.schemas.util.commonRestrictions

/**
 * Implementation of a schema representing enum values of type [T]. Use the [EnumSchema] constructor
 * function to create an instance of this class.
 *
 * @property kClass `KClass` of the enum represented by this schema.
 */
public open class EnumSchema<T : Enum<T>>
@PublishedApi
internal constructor(
    public val kClass: KClass<T>,
    validations: Iterable<Validation<T>> = emptyList(),
    override val initialValue: T,
) : AbstractSimpleSchema<T>(validations) {
    override val typeInfo: TypeInfo =
        TypeInfo(kClass, restrictions = commonRestrictions(validations))

    override fun assignableTo(type: KType): Boolean =
        (type.classifier as? KClass<*>)?.isInstance(initialValue) == true

    public companion object {
        // Since reified types in classes are not supported, we use a reified invoke function to
        // automatically provide the enum's `::class` and initial value.

        /** Function that returns a schema representing enum values of type [T]. */
        @JvmOverloads
        @JvmStatic
        @JvmName("create")
        public inline operator fun <reified T : Enum<T>> invoke(
            validations: Iterable<Validation<T>> = emptyList(),
            initialValue: T = enumEntries<T>().first(),
        ): EnumSchema<T> = EnumSchema(T::class, validations, initialValue)

        /** Function that returns a schema representing enum values of type [T]. */
        @JvmOverloads
        @JvmStatic
        @JvmName("create")
        public inline operator fun <reified T : Enum<T>> invoke(
            vararg validations: Validation<T>,
            initialValue: T = enumEntries<T>().first(),
        ): EnumSchema<T> = EnumSchema(validations.asIterable(), initialValue)
    }
}
